<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  job
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\Job\Collections;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

use E4JConnect\Pricelabs\Job\Collection;
use E4JConnect\Pricelabs\Job\Command;
use E4JConnect\Pricelabs\Job\Result;

/**
 * E4jConnect Pricelabs DatabaseCollection implementation.
 *
 * @since 1.0
 */
class DatabaseCollection implements Collection
{
    /** @var object[] */
    protected $queue = [];

    /** @var int */
    protected $queueId = 0;

    /** @var QueueModel */
    protected $queueModel;

    /** @var JobModel */
    protected $jobModel;

    /**
     * Class constructor will initiliaze the needed models.
     */
    public function __construct()
    {
        $this->queueModel = new \E4JConnect\Pricelabs\MVC\Models\QueueModel;
        $this->jobModel = new \E4JConnect\Pricelabs\MVC\Models\JobModel;
    }

    /**
     * @inheritDoc
     */
    public function createQueue(string $id, ?string $event = null)
    {
        $data = [
            'signature' => $id,
            'event'     => $event,
        ];

        $this->queueId = $this->queueModel->save($data);

        if (!$this->queueId)
        {
            $error = $this->queueModel->getError();

            if (!$error instanceof \Exception)
            {
                $error = new \RuntimeException($error ?: 'An error has occurred while creating the jobs queue.', 500);
            }

            throw $error;
        }
    }

    /**
     * @inheritDoc
     */
    public function push(Command $command)
    {
        $data = [
            'id_queue' => $this->queueId,
            'command'  => $command,
        ];

        $id = $this->jobModel->save($data);

        if (!$id)
        {
            $error = $this->jobModel->getError();

            if (!$error instanceof \Exception)
            {
                $error = new \RuntimeException($error ?: 'An error has occurred while scheduling the job.', 500);
            }

            throw $error;
        }

        $queue_item = $this->queueModel->getItem($this->queueId);

        $data = [
            'id'         => $this->queueId,
            'jobs_count' => $queue_item->jobs_count + 1,
        ];

        $this->queueModel->save($data);
    }

    /**
     * @inheritDoc
     */
    public function pull()
    {
        $job = $this->jobModel->getNextJob();

        if (!$job)
        {
            return null;
        }

        $this->queue[] = $job;

        $data = [
            'id'         => $job->id,
            'executedon' => \JFactory::getDate()->toSql(),
        ];

        $this->jobModel->save($data);

        return $job->command;
    }

    /**
     * @inheritDoc
     */
    public function complete(Command $command, Result $result)
    {
        if (($index = $this->findJob($command)) !== null)
        {
            $job = array_splice($this->queue, $index, 1);
            $job = $job[0];

            $result_status = 1;

            if ($result->getErrors())
            {
                $result_status = 0;
            }
            else if ($result->getWarnings())
            {
                $result_status = 2;
            }

            $data = [
                'id'          => $job->id,
                'status'      => $result_status,
                'results'     => $result,
                'completedon' => \JFactory::getDate()->toSql(),
            ];

            $this->jobModel->save($data);

            if ($job->id_queue)
            {
                $queue_item = $this->queueModel->getItem($job->id_queue);

                $data = [
                    'id'             => $job->id_queue,
                    'jobs_processed' => $queue_item->jobs_processed + 1,
                ];

                $this->queueModel->save($data);
            }
        }
    }

    /**
     * Finds the job scheduled by the provided command.
     * 
     * @param   Command  $command
     * 
     * @return  int|null
     */
    protected function findJob(Command $command)
    {
        foreach ($this->queue as $i => $job)
        {
            if ($job->command === $command)
            {
                return $i;
            }
        }

        return null;
    }
}
