<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  job
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\Job\Commands;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

use E4JConnect\Pricelabs\Job\Command;

/**
 * E4jConnect Pricelabs OptimizationCommand implementation.
 * Optimizes the database records for the newly created alteration rules.
 *
 * @since 2.2
 */
class OptimizationCommand implements Command
{
    /** @var array */
    protected $options;

    /**
     * Class constructor.
     * 
     * @param  array  $options  A configuration array holding the following attributes:
     *                          - room_id
     *                          - date_from
     *                          - date_to
     */
    public function __construct(array $options)
    {
        $this->options = $options;
    }

    /**
     * @inheritDoc
     */
    public function getSummary()
    {
        return __('Optimizing database records.', 'e4jconnect-pricelabs');
    }

    /**
     * @inheritDoc
     */
    public function getExtraData()
    {
        return sprintf(
            __('Optimized records from %s to %s (Listing #%d).', 'e4jconnect-pricelabs'),
            $this->options['date_from'] ?? '/',
            $this->options['date_to'] ?? '/',
            $this->options['room_id'] ?? '/'
        );
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $result = [
            'success' => [],
            'errors' => [],
        ];

        if (empty($this->options['room_id']))
        {
            $result['errors'][] = 'Missing room ID.';
        }

        if (empty($this->options['date_from']))
        {
            $result['errors'][] = 'Missing from date.';
        }

        if (empty($this->options['date_to']))
        {
            $result['errors'][] = 'Missing to date.';
        }

        if (empty($this->options['rate_id']))
        {
            $result['errors'][] = 'Missing rate plan ID.';
        }

        if ($result['errors'])
        {
            // abort because of missing required fields
            return new \E4JConnect\Pricelabs\Job\Results\JsonResult($result);
        }

        // optimize db records
        try {
            // inject options
            \VBOPerformanceCleaner::setOptions([
                'listing_id' => (int) $this->options['room_id'],
                'id_price'   => (int) $this->options['rate_id'],
                'from_date'  => $this->options['date_from'],
                'to_date'    => $this->options['date_to'],
            ]);

            // perform listing snapshot
            \VBOPerformanceCleaner::listingSeasonSnapshot();

            // register success
            $result['success'][] = __('Database records optimized.', 'e4jconnect-pricelabs');
        } catch (Throwable $e) {
            // register error
            $result['errors'][] = sprintf('Database records optimization failed (%d): %s', ($e->getCode() ?: 500), $e->getMessage());
        }

        return new \E4JConnect\Pricelabs\Job\Results\JsonResult($result);
    }
}
