<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  job
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\Job\Commands;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

use E4JConnect\Pricelabs\Job\Command;

/**
 * E4jConnect Pricelabs RARCommand implementation.
 *
 * @since 1.0
 */
class RARCommand implements Command
{
    /** @var array */
    protected $data;

    /** @var string */
    protected $apiUser;

    /**
     * Class constructor.
     * 
     * @param  array   $data
     * @param  string  $apiUser
     */
    public function __construct(array $data, ?string $apiUser = null)
    {
        $this->data    = $data;
        $this->apiUser = $apiUser;
    }

    /**
     * @inheritDoc
     */
    public function getSummary()
    {
        if ($this->data['date_from'] == $this->data['date_to'])
        {
            return sprintf(
                __("Update Room Rates, %s\n%s", 'e4jconnect-pricelabs'),
                \VikBooking::getCurrencySymb() . ' ' . \VikBooking::numberFormat($this->data['rates_data'][0]['cost']),
                $this->data['date_from']
            );
        }

        return sprintf(
            __("Update Room Rates, %s\n%s - %s", 'e4jconnect-pricelabs'),
            \VikBooking::getCurrencySymb() . ' ' . \VikBooking::numberFormat($this->data['rates_data'][0]['cost']),
            $this->data['date_from'],
            $this->data['date_to']
        );
    }

    /**
     * @inheritDoc
     */
    public function getExtraData()
    {
        // week days map
        $wdays_map = [
            __('Sun'),
            __('Mon'),
            __('Tue'),
            __('Wed'),
            __('Thu'),
            __('Fri'),
            __('Sat'),
        ];

        // get room data
        $room = \VikBooking::getRoomInfo($this->data['room_id'], ['id', 'name']);

        // build restrictions, if any
        $restrictions = [];

        if (isset($this->data['minlos']) && $this->data['minlos'] > 0)
        {
            $restrictions[] = sprintf(__('Min LOS %d', 'e4jconnect-pricelabs'), $this->data['minlos']);
        }

        if (isset($this->data['maxlos']) && $this->data['maxlos'] > 0)
        {
            $restrictions[] = sprintf(__('Max LOS %d', 'e4jconnect-pricelabs'), $this->data['maxlos']);
        }

        if (isset($this->data['cta']) && $this->data['cta'])
        {
            $restrictions[] = __('CTA', 'e4jconnect-pricelabs');
        }
        elseif (!empty($this->data['cta_wdays']))
        {
            $cta_wdays = array_map(function($wday) use ($wdays_map)
            {
                $index = array_search($wday, array_keys($wdays_map));

                if ($index !== false)
                {
                    return $wdays_map[$index];
                }

                return '';
            }, $this->data['cta_wdays']);

            $cta_wdays = array_filter($cta_wdays);

            if ($cta_wdays)
            {
                $restrictions[] = sprintf(__('CTA: %s', 'e4jconnect-pricelabs'), implode(', ', $cta_wdays));
            }
        }

        if (isset($this->data['ctd']) && $this->data['ctd'])
        {
            $restrictions[] = __('CTD', 'e4jconnect-pricelabs');
        }
        elseif (!empty($this->data['ctd_wdays']))
        {
            $ctd_wdays = array_map(function($wday) use ($wdays_map)
            {
                $index = array_search($wday, array_keys($wdays_map));

                if ($index !== false)
                {
                    return $wdays_map[$index];
                }

                return '';
            }, $this->data['ctd_wdays']);

            $ctd_wdays = array_filter($ctd_wdays);

            if ($ctd_wdays)
            {
                $restrictions[] = sprintf(__('CTD: %s', 'e4jconnect-pricelabs'), implode(', ', $ctd_wdays));
            }
        }

        return preg_replace("/,\s$/", '', sprintf("%s, %s, %s", $this->apiUser ?: '', $room['name'] ?? '', implode(", ", $restrictions)));
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        // obtain the execution results
        list($channels, $success, $warnings, $errors) = \VCMOtaRarUpdate::getInstance($this->data, $anew = true)
            // set the caller to App to reduce the sleep time between the requests
            ->setCaller('App')
            // set the API user with the currently authenticated account email
            ->setApiUser($this->apiUser)
            // execute the request
            ->execute();

        $success = array_merge(
            [
                __('New rates applied to the booking engine.', 'e4jconnect-pricelabs'),
            ],
            (array) $success
        );

        return new \E4JConnect\Pricelabs\Job\Results\JsonResult([
            'channels' => (array) $channels,
            'success'  => (array) $success,
            'warnings' => (array) $warnings,
            'errors'   => (array) $errors,
        ]);
    }
}
