<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  wordpress
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\MVC\Models;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

// load dependencies
\JLoader::import('adapter.mvc.models.form');

/**
 * E4jConnect Pricelabs queue model.
 *
 * @since 1.0
 */
class QueueModel extends \JModelForm
{
    /**
     * @inheritDoc
     */
    public function getTable($name = '', $prefix = 'JTable', $options = [])
    {
        if (!$name)
        {
            $name = $this->getName();
        }

        return new \E4JConnect\Pricelabs\MVC\Tables\QueueTable(\JFactory::getDbo());
    }

    /**
     * Returns the summary of a queue.
     * 
     * @param   string  $queueId
     * 
     * @return  object
     * 
     * @throws  \DomainException
     */
    public function getSummary(string $queueId)
    {
        $queue = $this->getItem([
            'signature' => $queueId,
        ]);

        if (!$queue)
        {
            throw new \DomainException('Queue not found', 404);
        }

        $queue->tot_success  = 0;
        $queue->tot_warnings = 0;
        $queue->tot_errors   = 0;
        $queue->jobs         = [];

        $dbo = \JFactory::getDbo();

        $dbo->setQuery(
            $dbo->getQuery(true)
                ->select($dbo->qn([
                    'command',
                    'createdon',
                    'results',
                    'executedon',
                    'status',
                ]))
                ->from($dbo->qn('#__e4jconnect_pricelabs_jobs'))
                ->where($dbo->qn('id_queue') . ' = ' . (int) $queue->id)
                ->order($dbo->qn('id') . ' ASC')
        );

        foreach ($dbo->loadObjectList() as $job)
        {
            $job->command = unserialize((string) $job->command);

            if ($job->command instanceof \E4JConnect\Pricelabs\Job\Command)
            {
                $job->extraData = $job->command->getExtraData();
                $job->command   = $job->command->getSummary();
            }
            else
            {
                $job->extraData = '';
                $job->command   = '/';
            }

            $job->results = json_decode($job->results ?: '{}');

            $queue->jobs[] = $job;

            $queue->tot_success  += ($job->results->success ?? []) ? 1 : 0;
            $queue->tot_warnings += ($job->results->warnings ?? []) ? 1 : 0;
            $queue->tot_errors   += ($job->results->errors ?? []) ? 1 : 0;
        }

        return $queue;
    }

    /**
     * @inheritDoc
     */
    public function delete($ids)
    {
        // delete all the specified queues first
        $deleted = parent::delete($ids);

        if (!$deleted)
        {
            return false;
        }

        $db = \JFactory::getDbo();

        $db->setQuery(
            // take all the jobs that belong to the delete queues
            $db->getQuery(true)
                ->select($db->qn('id'))
                ->from($db->qn('#__e4jconnect_pricelabs_jobs'))
                ->where($db->qn('id_queue') . ' IN (' . implode(',', array_map('intval', $ids)) . ')')
        );

        // delete the children jobs too
        (new JobModel)->delete($db->loadColumn());

        return true;
    }

    /**
     * Aborts an ongoing queue and its related jobs.
     * 
     * @param   string  $queueId
     * 
     * @return  void
     */
    public function abort(string $queueId)
    {
        $dbo = \JFactory::getDbo();

        $queue = $this->getItem([
            'signature' => $queueId,
        ]);

        if (!$queue)
        {
            throw new \DomainException('Queue not found', 404);
        }

        // build the properties to update
        $updateQueue = new \stdClass;
        $updateQueue->id      = $queue->id;
        $updateQueue->aborted = 1;

        // set queue as aborted
        $this->save($updateQueue);

        // update jobs status as aborted (3)
        $dbo->setQuery(
            $dbo->getQuery(true)
                ->update($dbo->qn('#__e4jconnect_pricelabs_jobs'))
                ->set($dbo->qn('status') . ' = 3')
                ->where($dbo->qn('id_queue') . ' = ' . $queue->id)
                ->where($dbo->qn('status') . ' IS NULL')
        );

        $dbo->execute();
    }

    /**
     * Deletes all the queues (and the related jobs) with a creation date
     * older than the specified threshold (calculated on the current time).
     * 
     * Only the queues that are fully completed will be taken. This can be
     * accomplished by making sure that the number of processed jobs is equal
     * to the total count of assigned jobs.
     * 
     * @param   string  $threshold  A day-based notation to fetch the threshold.
     * 
     * @return  bool    True in case something has been deleted, false otherwise.
     */
    public function flush(string $threshold = '-1 month')
    {
        try
        {
            $threshold = \JFactory::getDate($threshold);
        }
        catch (\Exception $error)
        {
            $threshold = \JFactory::getDate('-1 month');
        }

        $db = \JFactory::getDbo();

        $db->setQuery(
            // take all the queues older than 
            $db->getQuery(true)
                ->select($db->qn('id'))
                ->from($db->qn('#__e4jconnect_pricelabs_queue'))
                ->where($db->qn('createdon') . ' < ' . $db->q($threshold->toSql()))
                ->andWhere([
                    $db->qn('jobs_count') . ' <= ' . $db->qn('jobs_processed'),
                    $db->qn('aborted') . ' = 1',
                ], 'OR')
        );

        // delete all the matching jobs
        return $this->delete($db->loadColumn());
    }
}
