<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  vikbooking
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\VikBooking\Widgets;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

/**
 * Installs the Pricelabs Queue admin-widget within VikBooking.
 */
class PricelabsQueueAdminWidget extends \VikBookingAdminWidget
{
    /** @var int */
    protected static $instance_counter = -1;

    /** @var int */
    protected $limit = 10;

	/**
     * Class constructor will define the widget name and identifier.
     */
    public function __construct()
    {
        // call parent constructor
        parent::__construct();

        $this->widgetName  = __('PriceLabs - Queue', 'e4jconnect-pricelabs');
        $this->widgetDescr = __('Displays the queues submitted through PriceLabs.', 'e4jconnect-pricelabs');
        $this->widgetId    = 'pricelabs_queue';

        // define widget and icon and style name
        $this->widgetIcon = '<i class="' . \VikBookingIcons::i('code') . '"></i>';
        $this->widgetStyleName = 'red';
    }

    /**
     * Loads a list of queues. Method called
     * by the widget through AJAX requests.
     * 
     * @return  array
     */
    public function loadQueues()
    {
        $app   = \JFactory::getApplication();
        $input = $app->input;

        $offset  = $input->getInt('offset', 0);
        $length  = $input->getInt('length', $this->limit);
        $wrapper = $input->getString('wrapper', '');

        // access the Queues model
        $queuesModel = new \E4JConnect\Pricelabs\MVC\Models\QueuesModel;

        // get the list of queues
        $queues = $queuesModel->getList([],
            [
                'start' => $offset,
                'limit' => $length,
            ]
        );

        // count all records
        $totQueues = $queuesModel->getTotal();

        // check if a next page is available
        $has_next_page = ((count($queues) + $length * $offset) < $totQueues);

        // start output buffering
        ob_start();

        foreach ($queues as $queue)
        {
            $queueCompleted = $queue->jobs_processed >= $queue->jobs_count;

            $queueDate = \JFactory::getDate($queue->createdon, \JFactory::getApplication()->get('offset', 'UTC'));

            // queue status and label
            if ($queueCompleted)
            {
                $queueStatusCls = 'success';
                $queueStatusLbl = __('Completed', 'e4jconnect-pricelabs');
            }
            elseif ($queue->aborted)
            {
                $queueStatusCls = 'danger';
                $queueStatusLbl = __('Aborted', 'e4jconnect-pricelabs');
            }
            else
            {
                $queueStatusCls = 'warning';
                $queueStatusLbl = __('Ongoing', 'e4jconnect-pricelabs');
            }

            ?>
            <div class="vbo-widget-history-record" data-recordid="<?php echo $queue->id; ?>" onclick="vboWidgetPricelabsQueueOpenJobs('<?php echo $wrapper; ?>', '<?php echo $queue->signature; ?>');">
                <div class="vbo-widget-history-avatar">
                    <img class="vbo-widget-history-avatar-profile" src="<?php echo E4JCONNECT_PRICELABS_RESOURCES . 'images/pricelabs-logo.png'; ?>" />
                </div>
                <div class="vbo-widget-history-content">
                    <div class="vbo-widget-history-content-head">
                        <div class="vbo-widget-history-content-info-details">
                            <span class="label label-<?php echo $queueStatusCls; ?> vbo-status-label"><?php echo $queueStatusLbl; ?></span>
                            <h4><?php echo \JText::_($queue->event); ?></h4>
                        </div>
                        <div class="vbo-widget-history-content-info-booking">
                            <div class="vbo-widget-history-content-info-dates">
                                <span class="vbo-widget-history-date"><?php echo $queueDate->format('D, d M Y H:i', true); ?></span>
                            </div>
                            <div class="vbo-widget-history-booking-checkin">
                                <span><?php echo sprintf(__('Requests processed: %d / %d', 'e4jconnect-pricelabs'), $queue->jobs_processed, $queue->jobs_count); ?></span>
                            </div>
                        </div>
                    </div>
                    <div class="vbo-widget-history-content-info-msg">
                        <span class="vbo-widget-history-content-info-msg-descr"><?php echo $queue->signature; ?></span>
                    </div>
                </div>
            </div>
            <?php
        }

        // append navigation
        ?>
        <div class="vbo-widget-commands vbo-widget-commands-right">
            <div class="vbo-widget-commands-main">
            <?php
            if ($offset > 0)
            {
                // show backward navigation button
                ?>
                <div class="vbo-widget-command-chevron vbo-widget-command-prev">
                    <span class="vbo-widget-command-chevron-prev" onclick="vboWidgetPricelabsQueueNavigate('<?php echo $wrapper; ?>', -1);"><?php \VikBookingIcons::e('chevron-left'); ?></span>
                </div>
                <?php
            }
            if ($has_next_page)
            {
                // show forward navigation button
                ?>
                <div class="vbo-widget-command-chevron vbo-widget-command-next">
                    <span class="vbo-widget-command-chevron-next" onclick="vboWidgetPricelabsQueueNavigate('<?php echo $wrapper; ?>', 1);"><?php \VikBookingIcons::e('chevron-right'); ?></span>
                </div>
            <?php
            }
            ?>
            </div>
        </div>
        <?php

        // get the HTML buffer
        $html_content = ob_get_contents();
        ob_end_clean();

        // return an associative array of values
        return [
            'html'        => $html_content,
            'tot_records' => count($queues),
            'next_page'   => (int)$has_next_page,
        ];
    }

    /**
     * Loads a list of jobs for a given queue. Method
     * called by the widget through AJAX requests.
     * 
     * @return  array
     * 
     * @throws  \Exception
     */
    public function loadQueueJobs()
    {
        $app   = \JFactory::getApplication();
        $input = $app->input;

        $signature = $input->getString('signature', '');
        $wrapper   = $input->getString('wrapper', '');

        if (!$signature)
        {
            throw new \Exception(__('Missing queue signature', 'e4jconnect-pricelabs'), 400);
        }

        // access the Queues model
        $queuesModel = new \E4JConnect\Pricelabs\MVC\Models\QueuesModel;

        // get the requested queue
        $queue = $queuesModel->getList(
            [
                'search' => $signature,
            ],
            [
                'start' => 0,
                'limit' => 1,
            ]
        );

        if (!$queue)
        {
            throw new \Exception(__('Queue not found', 'e4jconnect-pricelabs'), 404);
        }

        $queue = $queue[0];

        // check if the queue is completed or still ongoing
        $queueCompleted = $queue->jobs_processed >= $queue->jobs_count;

        // get the queue summary
        $summary = (new \E4JConnect\Pricelabs\MVC\Models\QueueModel)->getSummary($queue->signature);

        // start output buffering
        ob_start();

        $tz_offset = \JFactory::getApplication()->get('offset', 'UTC');

        $queueDate = \JFactory::getDate($summary->createdon, $tz_offset);

        ?>
        <div class="vbo-admin-container vbo-admin-container-full vbo-admin-container-compact">
        	<div class="vbo-params-wrap">
				<div class="vbo-params-container">

					<div class="vbo-param-container">
						<div class="vbo-param-label">
                            <strong><?php echo \JText::_($summary->event); ?></strong>
                        </div>
						<div class="vbo-param-setting"><?php echo $queueDate->format('D, d M Y H:i', true); ?></div>
					</div>

					<div class="vbo-param-container">
						<div class="vbo-param-label"><?php echo __('Jobs', 'e4jconnect-pricelabs'); ?></div>
						<div class="vbo-param-setting">
							<span class="label label-info"><?php echo $summary->jobs_processed . ' / ' . $summary->jobs_count; ?></span>
						</div>
					</div>

					<div class="vbo-param-container">
						<div class="vbo-param-label"><?php echo __('Success', 'e4jconnect-pricelabs'); ?></div>
						<div class="vbo-param-setting">
							<span class="label label-success"><?php echo $summary->tot_success . ' / ' . $summary->jobs_count; ?></span>
						</div>
					</div>

					<div class="vbo-param-container">
						<div class="vbo-param-label"><?php echo __('Warnings', 'e4jconnect-pricelabs'); ?></div>
						<div class="vbo-param-setting">
							<span class="label label-warning"><?php echo $summary->tot_warnings . ' / ' . $summary->jobs_count; ?></span>
						</div>
					</div>

					<div class="vbo-param-container">
						<div class="vbo-param-label"><?php echo __('Errors', 'e4jconnect-pricelabs'); ?></div>
						<div class="vbo-param-setting">
							<span class="label label-error"><?php echo $summary->tot_errors . ' / ' . $summary->jobs_count; ?></span>
						</div>
					</div>

					<div class="vbo-param-container">
						<div class="vbo-param-label"><?php echo __('Jobs', 'e4jconnect-pricelabs'); ?></div>
                    <?php
                    if (!$queueCompleted && !$queue->aborted)
                    {
                        ?>
                        <div class="vbo-param-setting">
                            <button type="button" class="btn btn-small btn-danger" onclick="vboWidgetPricelabsQueueAbort('<?php echo $wrapper; ?>', '<?php echo $queue->signature; ?>');"><?php echo __('Abort queue', 'e4jconnect-pricelabs'); ?></span>
                        </div>
                        <?php
                    }
                    ?>
					</div>

				<?php
				// scan all jobs in the queue summary
				foreach ($summary->jobs as $job)
				{
					// job execution status icon
					if ($job->status == 1)
					{
						$job_exec_status_icn = \VikBookingIcons::i('check-circle');
					}
					elseif ($job->status == 2)
					{
						$job_exec_status_icn = \VikBookingIcons::i('exclamation-circle');
					}
					elseif ($job->status === 0 || $job->status === '0')
					{
						$job_exec_status_icn = \VikBookingIcons::i('times-circle');
					}
                    elseif ($job->status == 3)
                    {
                        $job_exec_status_icn = \VikBookingIcons::i('ban');
                    }
					else
					{
						$job_exec_status_icn = \VikBookingIcons::i('stopwatch');
					}

					?>
					<div class="vbo-param-container vbo-param-nested">
						<div class="vbo-param-label">
							<div>
								<i class="<?php echo $job_exec_status_icn; ?>"></i>
								<span><?php echo nl2br($job->command); ?></span>
							</div>
						</div>
						<div class="vbo-param-setting">
						<?php
						if ($job->extraData)
						{
							?>
							<div class="info" style="margin-top: 0;"><?php echo $job->extraData; ?></div>
							<?php
						}
						?>
                        <div class="pricelabs-jobs-results">
                        <?php
                        if ($job->results->success)
                        {
							?>
							<div class="label label-success"><?php echo implode(', ', array_values((array) $job->results->success)); ?></div>
							<?php
						}
						if ($job->results->warnings)
                        {
							?>
							<div class="label label-warning"><?php echo implode(', ', array_values((array) $job->results->warnings)); ?></div>
							<?php
						}
						if ($job->results->errors)
                        {
							?>
							<div class="label label-error"><?php echo implode(', ', array_values((array) $job->results->errors)); ?></div>
							<?php
						}
                        ?>
                        </div>
                        <?php
						if ($job->executedon)
						{
							?>
							<div class="vbo-param-setting-comment"><?php echo \JFactory::getDate($job->executedon, $tz_offset)->format('D, d M Y H:i', true); ?></div>
							<?php
						}
						?>
						</div>
					</div>
					<?php
				}
				?>

				</div>
			</div>
        </div>
        <?php

        // get the HTML buffer
        $html_content = ob_get_contents();
        ob_end_clean();

        // return an associative array of values
        return [
            'html' => $html_content,
        ];
    }

    /**
     * Aborts a given queue and its related jobs. Method
     * called by the widget through AJAX requests.
     * 
     * @return  array
     * 
     * @throws  \Exception
     */
    public function abortQueue()
    {
        $app   = \JFactory::getApplication();
        $input = $app->input;

        $queue_id = $input->getString('queue_id', '');

        if (!$queue_id)
        {
            throw new \Exception(__('Missing queue signature', 'e4jconnect-pricelabs'), 400);
        }

        // access the Queue model
        $queueModel = new \E4JConnect\Pricelabs\MVC\Models\QueueModel;

        // abort the queue
        $queueModel->abort($queue_id);

        // return an associative array of values
        return [
            'success' => $queue_id,
        ];
    }

    /**
     * @inheritDoc
     */
    public function render(?\VBOMultitaskData $data = null)
    {
        // increase widget's instance counter
        static::$instance_counter++;

        // check whether the widget is being rendered via AJAX when adding it through the customizer
        $is_ajax = $this->isAjaxRendering();

        // generate a unique ID for the wrapper instance
        $wrapper_instance = !$is_ajax ? static::$instance_counter : rand();
        $wrapper_id = 'vbo-widget-pricelabsqueue-' . $wrapper_instance;

        // get permissions
        $vbo_auth_pricing = \JFactory::getUser()->authorise('core.vbo.pricing', 'com_vikbooking');

        ?>

        <style type="text/css">
            .pricelabs-jobs-results > * {
                width: 100%;
                box-sizing: border-box;
                white-space: normal;
            }
        </style>

        <div id="<?php echo $wrapper_id; ?>" class="vbo-admin-widget-wrapper" data-offset="0" data-length="<?php echo $this->limit; ?>">

            <div class="vbo-admin-widget-head">
                <div class="vbo-admin-widget-head-inline">
                    <h4><?php echo $this->widgetIcon; ?> <span><?php echo $this->widgetName; ?></span></h4>

                    <div class="vbo-admin-widget-head-commands">
                        <div class="vbo-reportwidget-commands">
                            <div class="vbo-reportwidget-command-dots">
                                <span class="vbo-widget-command-togglefilters" onclick="vboWidgetPricelabsQueueNavigate('<?php echo $wrapper_id; ?>', 0);">
                                    <i class="vboicn-loop2"></i>
                                </span>
                            </div>
                        </div>
                    </div>

                </div>
            </div>

            <div class="vbo-widget-booskcal-mday-wrap vbo-widget-pricelabsqueue-jobs-wrap" data-queue-signature="" style="display: none;">
                <div class="vbo-widget-booskcal-mday-head">
                    <a class="vbo-widget-booskcal-mday-back" href="JavaScript: void(0);" onclick="vboWidgetPricelabsQueueJobsBack('<?php echo $wrapper_id; ?>');"><?php \VikBookingIcons::e('chevron-left'); ?> <?php echo __('Back', 'e4jconnect-pricelabs'); ?></a>
                </div>
                <div class="vbo-widget-pricelabsqueue-jobs-list"></div>
            </div>

            <div class="vbo-widget-latestevents-wrap vbo-widget-pricelabsqueue-wrap">
                <div class="vbo-widget-latestevents-inner vbo-widget-pricelabsqueue-inner">
                    <div class="vbo-widget-latestevents-list vbo-widget-pricelabsqueue-list">
                    <?php
                    for ($i = 0; $i < floor($this->limit / 2); $i++) {
                        ?>
                        <div class="vbo-dashboard-guest-activity vbo-dashboard-guest-activity-skeleton">
                            <div class="vbo-dashboard-guest-activity-avatar">
                                <div class="vbo-skeleton-loading vbo-skeleton-loading-avatar"></div>
                            </div>
                            <div class="vbo-dashboard-guest-activity-content">
                                <div class="vbo-dashboard-guest-activity-content-head">
                                    <div class="vbo-skeleton-loading vbo-skeleton-loading-title"></div>
                                </div>
                                <div class="vbo-dashboard-guest-activity-content-info-msg">
                                    <div class="vbo-skeleton-loading vbo-skeleton-loading-content"></div>
                                </div>
                            </div>
                        </div>
                        <?php
                    }
                    ?>
                    </div>
                </div>
            </div>

        </div>
        <?php

        if (static::$instance_counter === 0 || $is_ajax) {
            /**
             * Print the JS code only once for all instances of this widget.
             * The real rendering is made through AJAX, not when the page loads.
             */
            ?>
        <script type="text/javascript">

            /**
             * Display the loading skeletons.
             */
            function vboWidgetPricelabsQueueSkeletons(wrapper) {
                var widget_instance = jQuery('#' + wrapper);
                if (!widget_instance.length) {
                    return false;
                }
                widget_instance.find('.vbo-widget-pricelabsqueue-list').html('');
                for (var i = 0; i < <?php echo floor($this->limit / 2); ?>; i++) {
                    var skeleton = '';
                    skeleton += '<div class="vbo-dashboard-guest-activity vbo-dashboard-guest-activity-skeleton">';
                    skeleton += '   <div class="vbo-dashboard-guest-activity-avatar">';
                    skeleton += '       <div class="vbo-skeleton-loading vbo-skeleton-loading-avatar"></div>';
                    skeleton += '   </div>';
                    skeleton += '   <div class="vbo-dashboard-guest-activity-content">';
                    skeleton += '       <div class="vbo-dashboard-guest-activity-content-head">';
                    skeleton += '           <div class="vbo-skeleton-loading vbo-skeleton-loading-title"></div>';
                    skeleton += '       </div>';
                    skeleton += '       <div class="vbo-dashboard-guest-activity-content-info-msg">';
                    skeleton += '           <div class="vbo-skeleton-loading vbo-skeleton-loading-content"></div>';
                    skeleton += '       </div>';
                    skeleton += '   </div>';
                    skeleton += '</div>';
                    // append skeleton
                    jQuery(skeleton).appendTo(widget_instance.find('.vbo-widget-pricelabsqueue-list'));
                }
            }

            /**
             * Navigates back to the queues list from the queue jobs list.
             */
            function vboWidgetPricelabsQueueJobsBack(wrapper) {
                var widget_instance = jQuery('#' + wrapper);
                if (!widget_instance.length) {
                    return false;
                }

                // hide jobs wrapper
                widget_instance.find('.vbo-widget-pricelabsqueue-jobs-wrap').hide();

                // show queues list
                widget_instance.find('.vbo-widget-pricelabsqueue-list').show();

                // empty the list of jobs
                widget_instance.find('.vbo-widget-pricelabsqueue-jobs-list').html('');
            }

            /**
             * Loads the jobs for the given queue signature.
             */
            function vboWidgetPricelabsQueueOpenJobs(wrapper, signature) {
                var widget_instance = jQuery('#' + wrapper);
                if (!widget_instance.length) {
                    return false;
                }

                // hide queues list
                widget_instance.find('.vbo-widget-pricelabsqueue-list').hide();

                // empty the list of jobs by showing a loading icon
                widget_instance.find('.vbo-widget-pricelabsqueue-jobs-list').html('<?php echo \VikBookingIcons::e('circle-notch', 'fa-spin fa-fw'); ?>');

                // show jobs wrapper
                widget_instance.find('.vbo-widget-pricelabsqueue-jobs-wrap').show();

                // the widget method to call
                var call_method = 'loadQueueJobs';

                // make a request to load the queues
                VBOCore.doAjax(
                    "<?php echo $this->getExecWidgetAjaxUri(); ?>",
                    {
                        widget_id: "<?php echo $this->getIdentifier(); ?>",
                        call:      call_method,
                        return:    1,
                        signature: signature,
                        wrapper:   wrapper,
                    },
                    (response) => {
                        try {
                            var obj_res = typeof response === 'string' ? JSON.parse(response) : response;
                            if (!obj_res.hasOwnProperty(call_method) || !obj_res[call_method]) {
                                console.error('Unexpected JSON response', obj_res);
                                return false;
                            }

                            // set queue signature attribute
                            widget_instance.find('.vbo-widget-pricelabsqueue-jobs-wrap').attr('data-queue-signature', signature);

                            // replace HTML with the queue jobs list
                            widget_instance.find('.vbo-widget-pricelabsqueue-jobs-list').html(obj_res[call_method]['html']);
                        } catch(err) {
                            console.error('could not parse JSON response', err, response);
                        }
                    },
                    (error) => {
                        widget_instance.find('.vbo-widget-pricelabsqueue-jobs-list').html('');
                        alert(error.responseText);
                    }
                );
            }

            /**
             * Perform the request to load the queues.
             */
            function vboWidgetPricelabsQueueLoad(wrapper) {
                var widget_instance = jQuery('#' + wrapper);
                if (!widget_instance.length) {
                    return false;
                }

                // get vars for making the request
                var current_offset  = parseInt(widget_instance.attr('data-offset'));
                var length_per_page = parseInt(widget_instance.attr('data-length'));

                // the widget method to call
                var call_method = 'loadQueues';

                // make a request to load the queues
                VBOCore.doAjax(
                    "<?php echo $this->getExecWidgetAjaxUri(); ?>",
                    {
                        widget_id: "<?php echo $this->getIdentifier(); ?>",
                        call:      call_method,
                        return:    1,
                        offset:    current_offset,
                        length:    length_per_page,
                        wrapper:   wrapper,
                    },
                    (response) => {
                        try {
                            var obj_res = typeof response === 'string' ? JSON.parse(response) : response;
                            if (!obj_res.hasOwnProperty(call_method) || !obj_res[call_method]) {
                                console.error('Unexpected JSON response', obj_res);
                                return false;
                            }

                            // replace HTML with new queues list
                            widget_instance.find('.vbo-widget-pricelabsqueue-list').html(obj_res[call_method]['html']);
                            
                            // check results
                            var tot_records = obj_res[call_method]['tot_records'] || 0;
                            if (!isNaN(tot_records) && parseInt(tot_records) < 1) {
                                // no results can indicate the offset is invalid or too high
                                if (!isNaN(current_offset) && parseInt(current_offset) > 0) {
                                    // reset offset to 0
                                    widget_instance.attr('data-offset', 0);
                                    // show loading skeletons
                                    vboWidgetPricelabsQueueSkeletons(wrapper);
                                    // reload the first page
                                    vboWidgetPricelabsQueueLoad(wrapper);
                                }
                            }
                        } catch(err) {
                            console.error('could not parse JSON response', err, response);
                        }
                    },
                    (error) => {
                        // remove the skeleton loading
                        widget_instance.find('.vbo-widget-pricelabsqueue-list').find('.vbo-dashboard-guest-activity-skeleton').remove();
                        console.error(error);
                    }
                );
            }

            /**
             * Aborts the given queue.
             */
            function vboWidgetPricelabsQueueAbort(wrapper, queue_id) {
                var widget_instance = jQuery('#' + wrapper);
                if (!widget_instance.length) {
                    return false;
                }

                if (!queue_id) {
                    throw new Error('Missing queue ID');
                }

                // go back to the list of queues
                vboWidgetPricelabsQueueJobsBack(wrapper);

                // show loading skeletons
                vboWidgetPricelabsQueueSkeletons(wrapper);

                // the widget method to call
                var call_method = 'abortQueue';

                // make a request to load the queues
                VBOCore.doAjax(
                    "<?php echo $this->getExecWidgetAjaxUri(); ?>",
                    {
                        widget_id: "<?php echo $this->getIdentifier(); ?>",
                        call:      call_method,
                        return:    1,
                        queue_id:  queue_id,
                        wrapper:   wrapper,
                    },
                    (response) => {
                        try {
                            var obj_res = typeof response === 'string' ? JSON.parse(response) : response;
                            if (!obj_res.hasOwnProperty(call_method) || !obj_res[call_method]) {
                                console.error('Unexpected JSON response', obj_res);
                                return false;
                            }

                            // reload the first page of queues
                            vboWidgetPricelabsQueueLoad(wrapper);
                        } catch(err) {
                            console.error('could not parse JSON response', err, response);
                        }
                    },
                    (error) => {
                        // remove the skeleton loading
                        widget_instance.find('.vbo-widget-pricelabsqueue-list').find('.vbo-dashboard-guest-activity-skeleton').remove();
                        alert(error.responseText);
                    }
                );
            }

            /**
             * Navigate between the various pages of the queues, or simply reloads.
             */
            function vboWidgetPricelabsQueueNavigate(wrapper, direction) {
                var widget_instance = jQuery('#' + wrapper);
                if (!widget_instance.length) {
                    return false;
                }

                // show loading skeletons
                vboWidgetPricelabsQueueSkeletons(wrapper);

                // current offset
                var current_offset = parseInt(widget_instance.attr('data-offset'));

                // records per page
                var steps = <?php echo $this->limit; ?>;

                // check direction and update offsets for nav
                if (direction > 0) {
                    // navigate forward
                    widget_instance.attr('data-offset', (current_offset + steps));
                } else if (direction < 0) {
                    // navigate backward
                    var new_offset = current_offset - steps;
                    new_offset = new_offset >= 0 ? new_offset : 0;
                    widget_instance.attr('data-offset', new_offset);
                } else if (direction === 0) {
                    // check if one queue jobs should be reloaded, beside the queues
                    if (widget_instance.find('.vbo-widget-pricelabsqueue-jobs-wrap').is(':visible')) {
                        var queue_signature = widget_instance.find('.vbo-widget-pricelabsqueue-jobs-wrap').attr('data-queue-signature');
                        if (queue_signature) {
                            // reload the jobs of the current queue
                            vboWidgetPricelabsQueueOpenJobs(wrapper, queue_signature);
                        }
                    }
                }

                // launch navigation
                vboWidgetPricelabsQueueLoad(wrapper);
            }
            
        </script>
            <?php
        }
        ?>

        <script type="text/javascript">

            jQuery(function() {

                // when document is ready, load queues for this widget's instance
                vboWidgetPricelabsQueueLoad('<?php echo $wrapper_id; ?>');

            });
            
        </script>

        <?php
    }
}
