<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  wordpress
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\WordPress\Update;

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

\JLoader::import('adapter.acl.access');
\JLoader::import('adapter.database.helper');

/**
 * Class used to handle the upgrade of the plugin.
 *
 * @since 1.0
 */
class Manager
{
    /**
     * Checks if the current version should be updated.
     *
     * @param   string  $version  The version to check.
     *
     * @return  bool    True if should be updated, otherwise false.
     */
    public static function shouldUpdate($version)
    {
        if (is_null($version))
        {
            return false;
        }

        return version_compare($version, PRICELABS_E4JCONNECT_VIKBOOKING_SOFTWARE_VERSION, '<');
    }

    /**
     * Executes the SQL file for the installation of the plugin.
     *
     * @return  void
     */
    public static function install()
    {
        self::execSqlFile(E4JCONNECT_PRICELABS_BASE . DIRECTORY_SEPARATOR . 'sql' . DIRECTORY_SEPARATOR . 'install.mysql.utf8.sql');
        self::installAcl();
    }

    /**
     * Executes the SQL file for the uninstallation of the plugin.
     *
     * @return  void
     */
    public static function uninstall()
    {
        self::execSqlFile(E4JCONNECT_PRICELABS_BASE . DIRECTORY_SEPARATOR . 'sql' . DIRECTORY_SEPARATOR . 'uninstall.mysql.utf8.sql');
        self::uninstallAcl();
    }

    /**
     * Launches the process to finalise the update.
     *
     * @param   string  $version  The current version.
     * 
     * @return  bool    True on success, false otherwise.
     */
    public static function update($version)
    {
        $fixer = self::getFixer($version);

        // trigger before installation routine

        $res = $fixer->beforeInstallation();

        if ($res === false)
        {
            return false;
        }

        // install SQL statements

        $res = self::installSql($version);

        if ($res === false)
        {
            return false;
        }

        // install ACL

        $res = self::installAcl();

        if ($res === false)
        {
            return false;
        }

        // trigger after installation routine

        $res = $fixer->afterInstallation();

        return ($res === false ? false : true);
    }

    /**
     * Get the script class to run the installation methods.
     *
     * @param   string  $version  The current version.
     *
     * @return  E4JConnect\Pricelabs\WordPress\Update\Fixer
     */
    protected static function getFixer($version)
    {
        return new Fixer($version);
    }

    /**
     * Provides the installation of the ACL routines.
     *
     * @return  bool  True on success, otherwise false. 
     */
    protected static function installAcl()
    {
        $actions = \JAccess::getActions('pricelabs_e4jconnect_vikbooking');

        if (!$actions)
        {
            // nothing to install
            return true;
        }

        $roles = [
            get_role('administrator'),
        ];

        foreach ($roles as $role)
        {
            if ($role)
            {
                foreach ($actions as $action)
                {
                    $cap = \JAccess::adjustCapability($action->name, 'com_pricelabs_e4jconnect_vikbooking');
                    $role->add_cap($cap, true);
                }
            }
        }

        return true;
    }

    /**
     * Provides the uninstallation of the ACL routines.
     *
     * @return  bool  True on success, otherwise false. 
     */
    protected static function uninstallAcl()
    {
        $actions = \JAccess::getActions('pricelabs_e4jconnect_vikbooking');

        if (!$actions)
        {
            // nothing to uninstall
            return true;
        }

        $roles = [
            get_role('administrator'),
        ];

        foreach ($roles as $role)
        {
            if ($role)
            {
                foreach ($actions as $action)
                {
                    $cap = \JAccess::adjustCapability($action->name, 'com_pricelabs_e4jconnect_vikbooking');
                    $role->remove_cap($cap);
                }
            }
        }

        return true;
    }

    /**
     * Run all the proper SQL files.
     *
     * @param   string  $version  The current version.
     *
     * @return  bool    True on success, otherwise false.
     */
    protected static function installSql($version)
    {
        $db = \JFactory::getDbo();

        $ok = true;

        // fetch all the SQL installation files
        $files = \JFolder::files(E4JCONNECT_PRICELABS_BASE . '/sql/update/mysql', '\.sql$', $recursive = false, $fullPath = true);

        // make sure we have something to install
        if ($files)
        {
            try
            {
                foreach ($files as $file)
                {
                    $name = basename($file);
                    $sqlVersion = substr($name, 0, strrpos($name, '.'));

                    if (version_compare($sqlVersion, $version, '>'))
                    {
                        // in case the SQL version is newer, execute the queries listed in the file
                        self::execSqlFile($file, $db);
                    }
                }
            }
            catch (\Exception $e)
            {
                $ok = false;
            }
        }

        return $ok;
    }

    /**
     * Executes all the queries contained in the specified file.
     *
     * @param   string     $file  The SQL file to launch.
     * @param   JDatabase  $db    The database driver handler.
     *
     * @return  void
     */
    protected static function execSqlFile($file, $db = null)
    {
        // make sure the file to install exists
        if (!\JFile::exists($file))
        {
            return;
        }

        if ($db === null)
        {
            $db = \JFactory::getDbo();
        }

        $bytes = '';

        $handle = fopen($file, 'r');

        while (!feof($handle))
        {
            $bytes .= fread($handle, 8192);
        }

        fclose($handle);

        foreach (\JDatabaseHelper::splitSql($bytes) as $query)
        {
            $db->setQuery($query);
            $db->execute();
        }
    }
}
