<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  job
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\Job\Commands;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

use E4JConnect\Pricelabs\Job\Command;

/**
 * E4jConnect Pricelabs RARBulkCommand implementation.
 * Submits a bulk action to all the channels for a specific room on a given window.
 *
 * @since 2.0
 */
class RARBulkCommand implements Command
{
    /** @var array */
    protected $options;

    /**
     * Class constructor.
     * 
     * @param  array  $options  A configuration array holding the following attributes:
     *                          - room_id
     *                          - date_from
     *                          - date_to
     */
    public function __construct(array $options)
    {
        $this->options = $options;
    }

    /**
     * @inheritDoc
     */
    public function getSummary()
    {
        return __('Updating rates on the OTAs.', 'e4jconnect-pricelabs');
    }

    /**
     * @inheritDoc
     */
    public function getExtraData()
    {
        return sprintf(
            __('Sending bulk actions from %s to %s (Listing #%d).', 'e4jconnect-pricelabs'),
            $this->options['date_from'] ?? '/',
            $this->options['date_to'] ?? '/',
            $this->options['room_id'] ?? '/'
        );
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $result = [
            'success' => [],
            'errors' => [],
        ];

        if (empty($this->options['room_id']))
        {
            $result['errors'][] = 'Missing room ID.';
        }

        if (empty($this->options['date_from']))
        {
            $result['errors'][] = 'Missing from date.';
        }

        if (empty($this->options['date_to']))
        {
            $result['errors'][] = 'Missing to date.';
        }

        if ($result['errors'])
        {
            // abort because of missing required fields
            return new \E4JConnect\Pricelabs\Job\Results\JsonResult($result);
        }

        // notify the channels about the rates update
        $success = \VikChannelManager::autoBulkActions([
            'from_date' => $this->options['date_from'],
            'to_date' => $this->options['date_to'],
            'forced_rooms' => (int) $this->options['room_id'],
            'update' => 'rates',
        ]);

        if ($success)
        {
            $result['success'][] = __('Rates updated on OTAs. Check the results from the channel manager dashboard.', 'e4jconnect-pricelabs');
        }
        else
        {
            // cannot establish a connection with the servers
            $result['errors'][] = __('Could not update rates on OTAs. Launch a bulk action manually.', 'e4jconnect-pricelabs');
        }

        return new \E4JConnect\Pricelabs\Job\Results\JsonResult($result);
    }
}
