<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  job
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\Job;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

/**
 * E4jConnect Pricelabs Processor implementation.
 *
 * @since 1.0
 */
class Processor
{
    /** @var Collection */
    protected $collection;

    /** @var string */
    protected $queueId;

    /**
     * Class constructor.
     * 
     * @param  Collection  $collection  the interface used to write/read the jobs.
     */
    public function __construct(Collection $collection)
    {
        $this->collection = $collection;
    }

    /**
     * Registers the command for a queue.
     * 
     * @param   Command  $command
     * 
     * @return  self
     */
    public function register(Command $command)
    {
        if (!$this->queueId)
        {
            $this->queueId = md5(uniqid());

            $event = basename(str_replace('\\', '/', strtolower(get_class($command))));

            $this->collection->createQueue($this->queueId, $event);
        }

        $this->collection->push($command);

        return $this;
    }

    /**
     * Processes the pending jobs.
     * 
     * @param   int  $max  Maximum number of jobs to execute per flow.
     * 
     * @return  void
     */
    public function run(int $max = 5)
    {
        $i = 0;

        // Keep iterating as long as there's something to process.
        // Never exceed the provided threshold, unless a 0 was given.
        while ((++$i <= $max || !$max) && ($command = $this->collection->pull()))
        {
            try
            {
                /** @var Result */
                $result = $command->execute();
            }
            catch (\Throwable $error)
            {
                // an error has occurred, create a result placeholder
                $result = new \E4JConnect\Pricelabs\Job\Results\JsonResult([
                    'errors' => [
                        $error->getMessage(),
                    ],
                ]);
            }
            
            $this->collection->complete($command, $result);
        }
    }

    /**
     * Gets the current queue signature ID.
     * 
     * @return  string
     */
    public function getQueueID()
    {
        return $this->queueId;
    }
}
