<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  job
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\Job\Commands;

// No direct access to this file
defined('ABSPATH') or die('No script kiddies please!');

use E4JConnect\Pricelabs\Job\Command;
use E4JConnect\Pricelabs\Job\Result;

/**
 * This implementation aggregates a list of commands to simultaneously execute them at once.
 *
 * @since 2.0
 */
class GroupExecutionCommand implements Command
{
    /** @var Command[] */
    protected $commands = [];

    /**
     * Class constructor.
     * 
     * @param  Command[]  $commands
     */
    public function __construct(array $commands)
    {
        foreach ($commands as $command) 
        {
            if ($command instanceof Command)
            {
                $this->commands[] = $command;
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function getSummary()
    {
        return sprintf(
            __('Executing %d requests.', 'e4jconnect'),
            count($this->commands)
        );
    }

    /**
     * @inheritDoc
     */
    public function getExtraData()
    {
        $html = [];

        foreach ($this->commands as $i => $command)
        {
            $marginBottom = $i < count($this->commands) - 1 ? 10 : 0;

            $html[] = '<div class="group-exec-wrapper" style="margin-bottom: ' . $marginBottom . 'px;">'
                . '<div class="summary-note"><strong style="font-weight: 500;">' . $command->getSummary() . '</strong></div>'
                . '<div class="extra-note"><small style="font-style: italic;">' . $command->getExtraData() . '</small></div>'
                . '</div>';
        }

        return '<div class="group-exec-command">' . implode("\n", $html) . '</div>';
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $data = [];

        foreach ($this->commands as $command)
        {
            try
            {
                // do some stuff before executing the command
                $this->preflight($command);

                // obtain result
                $result = $command->execute();

                // do some stuff after executing the command
                $this->postflight($command, $result);

                // scan result properties
                foreach ($result->jsonSerialize() as $k => $v)
                {
                    if (in_array($k, ['success', 'warnings', 'errors']))
                    {
                        // join value
                        $data[$k] = array_values(array_unique(array_merge($data[$k] ?? [], $v)));
                    }
                    else
                    {
                        // replace value
                        $data[$k] = $v;
                    }
                }
            }
            catch (\Exception $e)
            {
                $data['warnings'] = array_merge($data['warnings'] ?? [], [$e->getMessage()]);
            }
        }

        // rebuild result object with all the merged details
        return new \E4JConnect\Pricelabs\Job\Results\JsonResult($data);
    }

    /**
     * Children classes that inherits this class can overwrite this method
     * to perform further actions before executing the current command.
     * 
     * @param   Command  $command  The command that will be executed after this method.
     * 
     * @return  void
     * 
     * @since   2.1
     */
    protected function preflight(Command $command)
    {
        // do nothing
    }

    /**
     * Children classes that inherits this class can overwrite this method
     * to perform further actions after executing the current command.
     * 
     * @param   Command  $command  The command that has been executed before this method.
     * @param   Result   $result   The response obtained with the command execution.
     * 
     * @return  void
     * 
     * @since   2.1
     */
    protected function postflight(Command $command, Result $result)
    {
        // do nothing
    }
}
