<?php
/** 
 * @package     Pricelabs E4jConnect-VikBooking
 * @subpackage  wordpress
 * @author      E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com | https://e4jconnect.com
 */

namespace E4JConnect\Pricelabs\WordPress\System;

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

use E4JConnect\Pricelabs\WordPress\Update\Manager as UpdateManager;

/**
 * Class used to handle the activation, deactivation and 
 * uninstallation of E4jConnect-Pricelabs plugin.
 *
 * @since 1.0
 */
class Installer
{
    /**
     * Flag used to init the class only once.
     *
     * @var bool
     */
    protected static $init = false;

    /**
     * Initialize the class attaching wp actions.
     *
     * @return  void
     */
    public static function onInit()
    {
        // init only if not done yet (back-end only)
        if (static::$init === false && is_admin())
        {
            // Attempt activation here because, in case VikBooking was not installed during the
            // activation, the database and ACL installation might have been missed.
            // Pass true to inform the system that we are manually attempting the activation.
            static::activate(true);

            // mark flag as true to avoid init it again
            static::$init = true;
        }
    }

    /**
     * Handles the activation of the plugin.
     * 
     * @param   bool  $silent  Flag used to check whether the activation has been 
     *                         triggered by WordPress (false) or not.
     *
     * @return  void
     */
    public static function activate(bool $silent = false)
    {
        // get installed software version
        $version = get_option('pricelabs_e4jconnect_vikbooking_software_version', null);

        // check if the plugin has been already installed
        if (is_null($version))
        {
            // dispatch UPDATER to launch installation queries
            UpdateManager::install();

            // mark the plugin has installed to avoid duplicated installation queries
            update_option('pricelabs_e4jconnect_vikbooking_software_version', PRICELABS_E4JCONNECT_VIKBOOKING_SOFTWARE_VERSION);

            // enqueue installation message
            \E4JConnect\Pricelabs\Core\Factory::getContainer()->get('system.messages')->success(
                '<p>Thanks for activating the <strong>Pricelabs E4jConnect-VikBooking</strong> plugin!</p>'
            );
        }
    }

    /**
     * Handles the deactivation of the plugin.
     *
     * @return  void
     */
    public static function deactivate()
    {

    }

    /**
     * Handles the uninstallation of the plugin.
     * 
     * IMPORTANT NOTE: the uninstallation does not run in case VikBooking has
     * been already deactivated or uninstalled.
     *
     * @return  void
     */
    public static function uninstall()
    {
        // dispatch UPDATER to drop database tables
        UpdateManager::uninstall();

        // delete installation flag
        delete_option('pricelabs_e4jconnect_vikbooking_software_version');
    }

    /**
     * Checks if the current version should be updated
     * and, eventually, processes it.
     * 
     * @return  void
     */
    public static function update()
    {
        // get installed software version
        $version = get_option('pricelabs_e4jconnect_vikbooking_software_version', null);

        // check if we are running an older version
        if (UpdateManager::shouldUpdate($version))
        {
            // process the update (we don't need to raise an error)
            UpdateManager::update($version);

            // update cached plugin version
            update_option('pricelabs_e4jconnect_vikbooking_software_version', PRICELABS_E4JCONNECT_VIKBOOKING_SOFTWARE_VERSION);
        }
    }
}
