<?php
/*
Plugin Name:  Pricelabs E4jConnect-VikBooking
Plugin URI:   https://vikwp.com/
Description:  Helper plugin to connect Pricelabs with E4jConnect-VikBooking.
Version:      2.3.1
Author:       E4J s.r.l.
Author URI:   https://vikwp.com
License:      GPL2
License URI:  https://www.gnu.org/licenses/gpl-2.0.html
Text Domain:  e4jconnect-pricelabs
Domain Path:  /languages
*/

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

// autoload dependencies
try
{
    require_once dirname(__FILE__) . DIRECTORY_SEPARATOR . 'autoload.php';
}
catch (RuntimeException $error)
{
    // something is not installed or is not active!
    add_action('admin_notices', function() use ($error)
    {
        ?>
        <div class="notice is-dismissible notice-warning">
            <p><?php echo $error->getMessage(); ?></p>
        </div>
        <?php
    });

    // return to avoid breaking the website
    return;
}

/**
 * Loads the plugin textdomain to support the i18n.
 */
add_action('init', function()
{
    load_plugin_textdomain('e4jconnect-pricelabs', false, dirname(plugin_basename(__FILE__)) . '/languages');

    // register the translations handler
    $language = \JFactory::getLanguage();
    $language->attachHandler(new \E4JConnect\Pricelabs\WordPress\Language\QueueEventHandler, 'e4jconnect-pricelabs');
});

// handle install/uninstall
register_activation_hook(__FILE__, ['E4JConnect\\Pricelabs\\WordPress\\System\\Installer', 'activate']);
register_deactivation_hook(__FILE__, ['E4JConnect\\Pricelabs\\WordPress\\System\\Installer', 'deactivate']);
register_uninstall_hook(__FILE__, ['E4JConnect\\Pricelabs\\WordPress\\System\\Installer', 'uninstall']);

//////////////////////////////////////
///// CHANNEL MANAGER API EVENTS /////
//////////////////////////////////////

/**
 * Filter to tell the API Framework if a custom request type is valid.
 * Use it to flag as valid a custom request type you are willing to implement.
 * 
 * @param  bool    $is_valid   True if the request type is what you are willing to implement, or false.
 * @param  string  $req_type  String that identifies the current request type.
 */
add_filter('vikchannelmanager_validate_app_request', function($is_valid, $req_type)
{
    return $is_valid || \E4JConnect\Pricelabs\Core\Factory::getContainer()->has("request.{$req_type}");
}, 10, 2);

/**
 * Filter to authorise the API Framework to execute a custom request type based on ACL settings in VCM.
 * Use it to restrict the access to specific custom request types you are implementing.
 * 
 * @param  array   $auth_rules  List of role and action to authorise for the user making the request.
 * @param  string  $req_type    String that identifies the current request type.
 * @param  string  $app_email   The login email of the App Framework user making the request.
 * @param  string  $role        The WordPress role of the user making the request.
 * @param  string  $action      The capability that the user making the request must have.
 */
add_filter('vikchannelmanager_authorise_app_request', function($auth_rules, $req_type, $app_email, $role, $action)
{
    if (!is_null($auth_rules)) {
        // another plugin must have authorised this request type already
        return $auth_rules;
    }

    try
    {
        $event = \E4JConnect\Pricelabs\Core\Factory::getContainer()->get("request.{$req_type}");
    }
    catch (\E4JConnect\Pricelabs\Psr\Container\NotFoundExceptionInterface $e)
    {
        return [$role, $action];
    }

    // ensure no duplicate actions have been registered
    remove_all_actions('vikchannelmanager_' . $req_type . '_app_request');

    /**
     * Triggers an action to let plugins execute a custom request type for the API Framework.
     * This is the "custom-named" hook name triggered by Vik Channel Manager with the E4jConnect APIs.
     * Use this hook to implement one custom request type with the related response.
     * 
     * @param   string  $req_type   string that identifies the current request type.
     * @param   object  $input      an instance of the VikWPInputJSON class to access request body vars.
     * @param   object  $response   the response object that will be sent to output as a JSON encoded string.
     * @param   string  $apiUser    the authenticated API user (email address) performing the request.
     * 
     * @return  void                nothing should be returned, but the $response "body" property should be set.
     */
    add_action('vikchannelmanager_' . $req_type . '_app_request', function($req_type, $input, $response, $apiUser) use ($event)
    {
        $event->dispatch($input, $response, $apiUser);
    }, 10, 4);

    return $event->getAccess($role, $action);
}, 10, 5);

/**
 * Triggers an action to let plugins manipulate the response for any request of the API Framework.
 * Use this hook only if you are willing to modify existing requests.
 * 
 * @param  string  $req_type    String that identifies the current request type.
 * @param  object  $input       An instance of the VikWPInputJSON class to access request body vars.
 * @param  object  $response    The response object that will be sent to output as a JSON encoded string.
 * @param  int     $statusCode  The HTTP response status code, defaults to 200.
 */
add_action('vikchannelmanager_completed_app_request', function($req_type, $input, $response, &$statusCode)
{
    if (!in_array($req_type, ['pricelabs_set_room_rates', 'pricelabs_async_queue_summary']))
    {
        // do not interfere with other request types
        return;
    }

    if ($statusCode !== 200 || is_scalar($response->body))
    {
        // an error must have occurred, do not manipulate the response object
        return;
    }

    // turn the response body object into the whole response object

    $body = $response->body;

    foreach ($response as $prop => $data)
    {
        unset($response->$prop);
    }

    foreach ($body as $prop => $data)
    {
        $response->$prop = $data;
    }
}, 10, 4);

///////////////////////////////////////
/////// VIKBOOKING ADMIN WIDGET ///////
///////////////////////////////////////

/**
 * Hook used to install the PriceLabs admin-widget within VikBooking.
 * 
 * @param  array  $return  An array containing the include paths of the widgets.
 */
add_filter('vikbooking_load_admin_widgets', function($return)
{
    if (!$return)
    {
        $return = [];
    }

    // alias the admin-widget class at runtime
    class_alias('E4JConnect\\Pricelabs\\VikBooking\\Widgets\\PricelabsQueueAdminWidget', 'VikBookingAdminWidgetPricelabsQueue');

    // install the admin-widget
    return array_merge(
        $return,
        [
            'pricelabs_queue',
        ]
    );
});

///////////////////////////////////////
//// VIKCHANNELMANAGER RATES FLOW /////
///////////////////////////////////////

/**
 * Hook used to save records to be stored in the rates flow table.
 * 
 * @param  bool   $return  Whether the record should be stored.
 * @param  array  $flow    An associative array for the flow record.
 * 
 * @since  2.1
 */
add_filter('vikchannelmanager_check_rates_flow_store_record', function($return, $flow)
{
    // store only PMS related records, skip OTA related records
    return ($flow['channel_id'] ?? -1) === -1 ? $return : false;
}, 10, 2);

///////////////////////////////////////
//////// PLUGIN UPDATE CHANNEL ////////
///////////////////////////////////////

/**
 * Handle the update process.
 * 
 * @since 2.0
 */
add_action('plugins_loaded', ['E4JConnect\\Pricelabs\\WordPress\\System\\Installer', 'update']);

/**
 * Subscribes the plugin to the update channel provided by VikUpdater 2.0.
 */
add_filter('vikupdater_subscribe_plugins', function($subscribers) {
    $subscribers[] = [
        // this is the SKU
        'sku' => 'e4jcpricelabs',
        // this is the plugin name (supports i18n)
        'name' => 'Pricelabs E4jConnect-VikBooking',
        // this is the base name of the plugin main file (without ".php")
        'slug' => 'e4jconnect-pricelabs',
        // this is usually built as "[FOLDER_NAME]/[MAIN_FILE_NAME].php"
        'plugin' => 'e4jconnect-pricelabs/pricelabs_e4jconnect_vikbooking.php',
        // the currently installed version of the plugin (possibly defined by a constant)
        'version' => PRICELABS_E4JCONNECT_VIKBOOKING_SOFTWARE_VERSION,
        // the minimum required version of WordPress (optional)
        'requires' => '4.0',
        // the minimum required version of PHP (optional)
        'requires_php' => '7.0',
        // the short description of the plugin (supports i18n)
        'description' => __('Helper plugin to connect Pricelabs with E4jConnect-VikBooking.', 'e4jconnect-pricelabs'),
        // specify the URL where the latest version of the plugin should be downloaded
        'download' => 'https://vikwp.com/api/?task=products.freedownload&sku=e4jcpricelabs',
    ];

    return $subscribers;
});

/**
 * Registers the program icons that will be used by WordPress when a new version is detected.
 */
add_filter('vikupdater_prepare_update_plugin_data', function($update, $manifest, $options) {
    // make sure we are observing our plugin (same as the "slug" previously registered)
    if ($update->slug === 'e4jconnect-pricelabs') {
        // register the plugin icons
        $update->icons = [
            '2x' => plugin_dir_url(__FILE__) . 'resources/images/logo-256x256.jpg',
            '1x' => plugin_dir_url(__FILE__) . 'resources/images/logo-128x128.jpg',
        ];
    }

    return $update;
}, 10, 3);

///////////////////////////////////////
///// PLUGIN CRON JOBS SCHEDULING /////
///////////////////////////////////////

/**
 * Filters the non-default cron schedules.
 * Adds support to repeat the cron jobs every minute.
 *
 * @param  array  $schedules  An array of non-default cron schedules.
 */
add_filter('cron_schedules', function($schedules)
{
    // add support for "every minute" recurrence
    $schedules['every_minute'] = array(
        'interval' => MINUTE_IN_SECONDS,
        'display'  => __('Every Minute', 'e4jconnect-pricelabs'),
    );

    return $schedules;
});

/**
 * Declare the callback that will be fired by WP-Cron to process the jobs queue.
 */
add_action('e4jconnect_pricelabs_cron_process_queue', function()
{
    \E4JConnect\Pricelabs\Core\Factory::getContainer()->get('job.processor')->run($max = 6);
});

/**
 * Declare the callback that will be fired by WP-Cron to flush the jobs queue.
 */
add_action('e4jconnect_pricelabs_cron_flush_queue', function()
{
    // delete all the completed queues older than 1 month (default argument)
    (new \E4JConnect\Pricelabs\MVC\Models\QueueModel)->flush(/* "-1 month" */);
});

/**
 * Install the scheduling of the hooks within WP-Cron to process the queue of jobs.
 * Priority (3rd) argument must be set to PHP_INT_MAX because Vik Booking uses (PHP_INT_MAX - 1)
 * to register all the interval schedules.
 */
add_action('plugins_loaded', function()
{
    if (!wp_next_scheduled('e4jconnect_pricelabs_cron_process_queue'))
    {
        // the "every_minute" interval is defined by this plugin
        wp_schedule_event(time(), 'every_minute', 'e4jconnect_pricelabs_cron_process_queue');
    }

    if (!wp_next_scheduled('e4jconnect_pricelabs_cron_flush_queue'))
    {
        // the "weekly" interval is native in WordPress
        wp_schedule_event(time(), 'weekly', 'e4jconnect_pricelabs_cron_flush_queue');
    }
}, PHP_INT_MAX);
