<?php
/*
Plugin Name:  White Label
Version:      1.0.1
License:      GPL2
License URI:  https://www.gnu.org/licenses/gpl-2.0.html
*/

/**
 * White Label Must-Use plugin global namespace.
 */
namespace VikWP\WhiteLabel {

    // No direct access
    defined('ABSPATH') or die('No script kiddies please!');

    /**
     * Define here the labels that will replace the original 
     * names of the VikPlugins.
     */
    const VAP_PLACEHOLDER = 'Appointments';
    const VBO_PLACEHOLDER = 'Bookings';
    const VRC_PLACEHOLDER = 'Rent a Car';
    const VRI_PLACEHOLDER = 'Rent Items';
    const VRE_PLACEHOLDER = 'Restaurant';

    /**
     * Define here the name and URL of the company that will be displayed
     * below the information of the Vik plugins.
     */
    const COMPANY_NAME_PLACEHOLDER = 'ACME';
    const COMPANY_URL_PLACEHOLDER  = 'https://acme.com';

}

/**
 * Namespace used to white-label any reference to our plugins as well as
 * to our company, such as VikWP or E4J srl.
 */
namespace VikWP\WhiteLabel\Hooks\Language {

    /**
     * Filters text with its translation.
     * 
     * Used to replace the original names of the Vik plugins with
     * the defined labels.
     *
     * @since  2.0.11
     *
     * @param  string  $translation  Translated text.
     * @param  string  $text         Text to translate.
     * @param  string  $domain       Text domain. Unique identifier for retrieving translated strings.
     */
    \add_filter('gettext', function($translation, $text, $domain) {
        $settingsModel = new \VikWP\WhiteLabel\Models\SettingsModel;

        $translation = preg_replace("/vik\s?appointments/i", $settingsModel->get('vikappointments.name', \VikWP\WhiteLabel\VAP_PLACEHOLDER), $translation);
        $translation = preg_replace(     "/vik\s?booking/i", $settingsModel->get(     'vikbooking.name', \VikWP\WhiteLabel\VBO_PLACEHOLDER), $translation);
        $translation = preg_replace(  "/vik\s?rent\s?car/i", $settingsModel->get(     'vikrentcar.name', \VikWP\WhiteLabel\VRC_PLACEHOLDER), $translation);
        $translation = preg_replace("/vik\s?rent\s?items/i", $settingsModel->get(   'vikrentitems.name', \VikWP\WhiteLabel\VRI_PLACEHOLDER), $translation);
        $translation = preg_replace( "/vik\s?restaurants/i", $settingsModel->get( 'vikrestaurants.name', \VikWP\WhiteLabel\VRE_PLACEHOLDER), $translation);

        $translation = preg_replace("/https:\/\/vikwp\.com/i", $settingsModel->get( 'company.url',  \VikWP\WhiteLabel\COMPANY_URL_PLACEHOLDER), $translation);
        $translation = preg_replace(       "/E4J s.?r.?l.?/i", $settingsModel->get('company.name', \VikWP\WhiteLabel\COMPANY_NAME_PLACEHOLDER), $translation);
        $translation = preg_replace(       "/VikWP(\.com)?/i", $settingsModel->get('company.name', \VikWP\WhiteLabel\COMPANY_NAME_PLACEHOLDER), $translation);

        return $translation;
    }, 100, 3);

}

/**
 * Namespace used to unset the "help" tab from the WP screen, as it contains clear references
 * to VikWP, since it is built by downloading the documentation from our website.
 */
namespace VikWP\WhiteLabel\Hooks\Help {

    /**
     * Fires after the current screen has been set.
     * 
     * Used to unset the "Help" tab from the WP screen options.
     *
     * @since  3.0.0
     *
     * @param  WP_Screen  $screen  Current WP_Screen object.
     */
    \add_action('current_screen', function($screen) {
        if (!$screen instanceof \WP_Screen) {
            // screen not yet ready, do nothing
            return;
        }

        // iterate all the registered tabs
        foreach ($screen->get_help_tabs() as $tabId => $tab) {
            if (preg_match("/^vik(appointments|restaurants)/", $tabId)) {
                // remove tab registered by VikAppointments
                $screen->remove_help_tab($tabId);
            }
        }
    }, 100);

}

/**
 * Namespace used to hide the license information from the plugin, such as the
 * "PRO Version" button and the footer (setting and text).
 */
namespace VikWP\WhiteLabel\Hooks\License {

    /**
     * Prints scripts and styles within the admin footer.
     * 
     * Used to hide the "PRO Version" buttons and company related settings.
     * 
     * @since 1.0
     */
    \add_action('admin_footer', function() {
        // hide "PRO Version" button from the main menu of VikAppointments and VikRestaurants
        echo '<style>.license-box.custom.is-pro { display: none !important; }</style>';
        // hide "PRO Version" button from the main menu of VikBooking
        echo '<style>.vbo-gotopro, .vbo-alreadypro { display: none !important; }</style>';
        // hide "PRO Version" button from the main menu of VikRentCar
        echo '<style>.vrc-gotopro, .vrc-alreadypro { display: none !important; }</style>';
        // hide "PRO Version" button from the main menu of VikRentItems
        echo '<style>.vri-gotopro, .vri-alreadypro { display: none !important; }</style>';

        // manually hide the plugins footer
        echo '<style>#vapfooter, #hmfooter, #vrestfooter { display: none !important; } </style>';

        // manually hide the "pro-only" links on VikBooking
        echo '<style>.vbo-menu-vcm-only { display: none !important; }</style>';
        ?>
    <script>
        (function($) {
            'use strict';

            $(function() {
                // remove the "Show Footer" setting from the configuration of the VIK plugins
                $('input[name="showfooter"]').closest('.control').hide();
                $('input[name="showfooter"]').closest('.control-group').hide();
                $('input[name="showfooter"]').closest('.vbo-param-container').hide();
                $('input[name="showfooter"]').closest('.vrc-param-container').hide();
                $('input[name="showfooter"]').closest('.vri-param-container').hide();
            });
        })(jQuery);
    </script>
        <?php
    });

    /**
     * Prevent the system from displaying advertising banners on certain widgets in
     * case VikChannelManager is not installed on the website.
     * 
     * @since 1.0.1
     */
    \add_filter('vikbooking_preflight_admin_widget', function($status, $widget) {
        if (in_array($widget, ['aitools', 'guest_messages', 'guest_reviews', 'latest_from_guests', 'operators_chat'])) {
            $status = class_exists('VikChannelManager');
        }

        return $status;
    }, 10, 2);

}

/**
 * Namespace used to turn off the RSS feature.
 */
namespace VikWP\WhiteLabel\Hooks\Rss {

    /**
     * Helper function used to disable the RSS feature from all the VIK plugins.
     * 
     * @param  JRssReader  &$rss  The RSS reader handler.
     */
    function disable_rss_feature($rss)
    {
        try {
            if ($rss->optedIn()) {
                // throw an exception in case the user already opted-in the RSS service
                throw new \Exception;
            }
        } catch (\Exception $error) {
            // manually disable the RSS service
            $rss->optIn(false);
        }
    }

    /**
     * Hook used to apply some stuff before returning the RSS reader.
     *
     * @param  JRssReader  &$rss  The RSS reader handler.
     */
    \add_action('vikappointments_before_use_rss', 'VikWP\\WhiteLabel\\Hooks\\Rss\\disable_rss_feature', 100);
    \add_action(     'vikbooking_before_use_rss', 'VikWP\\WhiteLabel\\Hooks\\Rss\\disable_rss_feature', 100);
    \add_action(     'vikrentcar_before_use_rss', 'VikWP\\WhiteLabel\\Hooks\\Rss\\disable_rss_feature', 100);
    \add_action(   'vikrentitems_before_use_rss', 'VikWP\\WhiteLabel\\Hooks\\Rss\\disable_rss_feature', 100);
    \add_action( 'vikrestaurants_before_use_rss', 'VikWP\\WhiteLabel\\Hooks\\Rss\\disable_rss_feature', 100);

    /**
     * Helper function used to remove the RSS fieldset from the configuration of the VIK plugins.
     * 
     * @param  array  $forms  The configuration forms.
     */
    function unset_rss_configuration($forms)
    {
        unset($forms['RSS']);

        return $forms;
    }

    /**
     * Unset RSS fieldset from the global configuration of VikBooking because there is no
     * way to hide it via CSS because of a unique identifier.
     * 
     * @param  array  $forms  The custom fieldsets.
     */
    \add_filter('vikappointments_display_view_config_global', 'VikWP\\WhiteLabel\\Hooks\\Rss\\unset_rss_configuration', 100);
    \add_filter(     'vikbooking_display_view_config_global', 'VikWP\\WhiteLabel\\Hooks\\Rss\\unset_rss_configuration', 100);
    \add_filter(     'vikrentcar_display_view_config_global', 'VikWP\\WhiteLabel\\Hooks\\Rss\\unset_rss_configuration', 100);
    \add_filter(   'vikrentitems_display_view_config_global', 'VikWP\\WhiteLabel\\Hooks\\Rss\\unset_rss_configuration', 100);
    \add_filter( 'vikrestaurants_display_view_config_global', 'VikWP\\WhiteLabel\\Hooks\\Rss\\unset_rss_configuration', 100);

    /**
     * Fires after core widgets for the admin dashboard have been registered.
     * 
     * Manually detaches the RSS widgets from the WordPress admin dashboard.
     *
     * @since 2.5.0
     */
    \add_action('wp_dashboard_setup', function() {
        global $wp_meta_boxes;

        // iterate all the supported VIK plugins
        foreach (['vikappointments', 'vikbooking', 'vikrentcar', 'vikrentitems', 'vikrestaurants'] as $slug) {
            // rewrite the widget ID (e.g. vik_appointments_rss)
            $widgetId = preg_replace("/^vik/", 'vik_', $slug) . '_rss';

            // check whether the dashboard supports a RSS widget for the current plugin
            if (isset($wp_meta_boxes['dashboard']['normal']['core'][$widgetId])) {
                // manually unregister the widget from the global WP meta boxes
                unset($wp_meta_boxes['dashboard']['normal']['core'][$widgetId]);
            }
        }
    }, 100);

}

/**
 * Namespace used to alter the normal behavior of the plugin links.
 * We need to disable the feedback feature as well as link to access the plugin information
 * hosted on wordpress.org.
 */
namespace VikWP\WhiteLabel\Hooks\PluginLinks {

    /**
     * Filters the array of row meta for each plugin in the Plugins list table.
     *
     * @since  2.8.0
     *
     * @param  string[]  $plugin_meta  An array of the plugin's metadata, including
     *                                 the version, author, author URI, and plugin URI.
     * @param  string    $plugin_file  Path to the plugin file relative to the plugins directory.
     * @param  array     $plugin_data  An array of plugin data.
     * @param  string    $status       Status filter currently applied to the plugin list.
     */
    \add_filter('plugin_row_meta', function($plugin_meta, $plugin_file, $plugin_data, $status) {
        $companyUrl = (new \VikWP\WhiteLabel\Models\SettingsModel)->get('company.url');

        // make sure we are observing a VIK plugin
        if (in_array($plugin_data['slug'] ?? '', ['vikappointments', 'vikbooking', 'vikrentcar', 'vikrentitems', 'vikrestaurants'])) {
            foreach ($plugin_meta as $k => $str) {
                // check if we have a link to show the information of the plugin from WordPress.org (View Details)
                if (stripos($str, 'plugin-information') !== false) {
                    // disable link
                    unset($plugin_meta[$k]);

                    if ($companyUrl) {
                        $aria_label = sprintf(
                            \__( 'Visit plugin site for %s'),
                            $plugin_data['Name']
                        );

                        // register a link to access the plugin author URI
                        $plugin_meta[] = sprintf(
                            '<a href="%s" aria-label="%s">%s</a>',
                            \esc_url( $companyUrl ),
                            \esc_attr( $aria_label ),
                            \__( 'Visit plugin site' )
                        );
                    }
                }
            }
        }

        return $plugin_meta;
    }, 100, 4);

    /**
     * Filters the action links displayed for each plugin in the Plugins list table.
     *
     * @since  2.5.0
     * @since  2.6.0  The `$context` parameter was added.
     * @since  4.9.0  The 'Edit' link was removed from the list of action links.
     *
     * @param string[]  $actions      An array of plugin action links. By default this can include
     *                                'activate', 'deactivate', and 'delete'. With Multisite active
     *                                this can also include 'network_active' and 'network_only' items.
     * @param string    $plugin_file  Path to the plugin file relative to the plugins directory.
     * @param array     $plugin_data  An array of plugin data. See get_plugin_data()
     *                                and the {@see 'plugin_row_meta'} filter for the list
     *                                of possible values.
     * @param string    $context      The plugin context. By default this can include 'all',
     *                                'active', 'inactive', 'recently_activated', 'upgrade',
     *                                'mustuse', 'dropins', and 'search'.
     */
    \add_filter('plugin_action_links', function($actions, $plugin_file, $plugin_data, $context) {
        // make sure we are observing a VIK plugin
        if (in_array($plugin_data['slug'] ?? '', ['vikappointments', 'vikbooking', 'vikrentcar', 'vikrentitems', 'vikrestaurants'])) {
            // Register a cookie to skip the feedback request.
            // Even if the headers have been already sent, the system will behave as expected because
            // the cookie value is still saved within our internal storage.
            \JFactory::getApplication()->input->cookie->set($plugin_data['slug'] . '_feedback', 1);
        } else if ($plugin_file === 'whitelabel.php') {
            // take the ID of the user that should be able to access the settings of the plugin
            $managerId = \get_option('whitelabel_manager_user_id');

            // proceed only in case the current user is authorized to manage the settings of this plugin
            if ($managerId == \wp_get_current_user()->ID) {
                // register the link to access the configuration of the White Label plugin
                $actions[] = sprintf(
                    '<a href="%s" class="thickbox" aria-label="%s">%s</a>',
                    \esc_url( '#TB_inline?width=500&height=400&inlineId=whitelabel-settings' ),
                    \esc_attr( 'Edit White Label preferences' ),
                    \__( 'Settings' )
                );

                // append thickbox to admin footer
                \add_action('admin_footer', function() {
                    (new \VikWP\WhiteLabel\Views\SettingsThickbox)->render();
                });   
            }
        }

        return $actions;
    }, 10, 4);

    /**
     * Filters the array of plugins for the list table.
     *
     * @since  6.3.0
     *
     * @param  array[]  $plugins  An array of arrays of plugin data, keyed by context.
     */
    add_filter('plugins_list', function($plugins) {
        $settingsModel = new \VikWP\WhiteLabel\Models\SettingsModel;

        // fetch the name of the "White Label" plugin from its configuration
        $pluginName = $settingsModel->get('company.plugin.name');

        if ($pluginName) {
            // overwrite the name and the title of this plugin with the specified one
            $plugins['mustuse']['whitelabel.php']['Name'] = $plugins['mustuse']['whitelabel.php']['Title'] = $pluginName;
        }   

        // fetch the description of the "White Label" plugin from its configuration
        $pluginDesc = $settingsModel->get('company.plugin.description');

        if ($pluginDesc) {
            // overwrite the description of this plugin with the specified one
            $plugins['mustuse']['whitelabel.php']['Description'] = $pluginDesc;
        }        

        return $plugins;
    });
}

/**
 * Namespace used to safely disable the WordPress update features.
 */
namespace VikWP\WhiteLabel\Hooks\PluginUpdate {

    /**
     * Filters the value of an existing site transient.
     *
     * Disable the updates for the VIK plugins.
     *
     * @since  2.9.0
     * @since  4.4.0  The `$transient` parameter was added.
     *
     * @param  mixed  $value     Value of site transient.
     * @param  string $transient Transient name.
     */
    \add_filter('site_transient_update_plugins', function($updates) {
        $settingsModel = new \VikWP\WhiteLabel\Models\SettingsModel;

        // iterate all the supported VIK plugins
        foreach (['vikappointments', 'vikbooking', 'vikrentcar', 'vikrentitems', 'vikrestaurants'] as $slug) {
            $pluginId = $slug . '/' . $slug . '.php';

            // check if we have an update available for this plugin
            if (!isset($updates->response[$pluginId])) {
                continue;
            }

            // check whether the updates should be allowed for this plugin
            if ($settingsModel->get($slug . '.updates', false)) {
                continue;
            }

            // revert the version to the current one
            $data = $updates->response[$pluginId];
            $data->new_version = \get_option($slug . '_software_version') ?: $data->new_version;

            // unset plugin from the list
            unset($updates->response[$pluginId]);

            if (!isset($updates->no_update)) {
                $updates->no_update = [];
            }

            // relocate the plugin within the "no updates available" list
            $updates->no_update[$pluginId] = $data;
        }

        return $updates;
    }, 100);

    /**
     * Filters whether to automatically update core, a plugin, a theme, or a language.
     *
     * Since WordPress 3.7, minor and development versions of core, and translations have
     * been auto-updated by default. New installs on WordPress 5.6 or higher will also
     * auto-update major versions by default. Starting in 5.6, older sites can opt-in to
     * major version auto-updates, and auto-updates for plugins and themes.
     *
     * See the {@see 'allow_dev_auto_core_updates'}, {@see 'allow_minor_auto_core_updates'},
     * and {@see 'allow_major_auto_core_updates'} filters for a more straightforward way to
     * adjust core updates.
     *
     * @since  3.7.0
     * @since  5.5.0  The `$update` parameter accepts the value of null.
     *
     * @param  bool|null  $update  Whether to update. The value of null is internally used
     *                             to detect whether nothing has hooked into this filter.
     * @param  object     $item    The update offer.
     */
    \add_filter('auto_update_plugin', function($update, $item) {
        // make sure we are observing a VIK plugin
        if (in_array($item->slug ?? '', ['vikappointments', 'vikbooking', 'vikrentcar', 'vikrentitems', 'vikrestaurants'])) {
            // disable auto-updates
            return false;
        }

        return $update;
    }, 100, 2);

}

/**
 * Namespace used to deal with the rendering of forms and fields.
 */
namespace VikWP\WhiteLabel\Form {

    /**
     * Form fieldset renderer class.
     */
    class FormFieldset
    {
        /** @var string */
        protected string $legend;

        /** @var FormField[] */
        protected array $fields = [];

        /**
         * Class constructor.
         * 
         * @param  string       $legend  An optional legend to display at the beginning of the fieldset.
         * @param  FormField[]  $fields  A list of fields to display within the body of the fieldset.
         */
        public function __construct(string $legend, array $fields = [])
        {
            $this->legend = $legend;

            foreach ($fields as $field) {
                if ($field instanceof FormField) {
                    $this->addField($field);
                }
            }
        }

        /**
         * Adds a new field.
         * 
         * @param   FormField  $field
         * 
         * @return  static
         */
        final public function addField(FormField $field)
        {
            $this->fields[] = $field;

            return $this;
        }

        /**
         * Renders the form fieldset.
         * 
         * @return  string
         */
        public function render()
        {
            // add legend, only if not empty
            if ($this->legend) {
                $legend = '<h2>' . $this->legend . '</h2>';
            }

            // create form fieldset
            return '<div class="whitlabel-settings-fieldset company">' . ($legend ?? '') . '<div class="form-wrap">' . implode("\n", $this->fields) . '</div>' . '</div>';
        }

        /**
         * @inheritDoc
         */
        #[\ReturnTypeWillChange]
        final public function __toString()
        {
            return $this->render();
        }
    }

    /**
     * Form field renderer class.
     */
    abstract class FormField
    {
        /**
         * Static counter used to avoid elements with duplicate IDs.
         * 
         * @var int
         */
        protected static $increment = 0;

        /** @var array */
        protected array $args;

        /** @var string */
        protected string $type;

        /**
         * Class constructor.
         * 
         * @param  array  $args  An associative array defining the field data.
         */
        public function __construct(array $args)
        {
            $this->args = $args;

            $name = (string) ($this->args['name'] ?? '');

            // auto-generate an ID starting from the provided name
            if (empty($this->args['id']) && !empty($name)) {
                // auto-generate ID from name
                $id = preg_replace("/[^a-zA-Z0-9\-_]+/", '_', $name);

                if (!empty($this->args['multiple']) || preg_match("/\[\]$/", $name))
                {
                    // in case of multiple field, append an incremental number to the ID
                    $id .= '_' . (++$this->$increment);
                }

                if ($id)
                {
                    $this->args['id'] = rtrim($id, '_');
                }
            }

            if (!empty($this->args['multiple']) && $name && !preg_match("/\[\]$/", $name))
            {
                // in case we have a multiple field and the name does not end with "[]", append it
                $this->args['name'] .= '[]';
            }
        }

        /**
         * Renders the form field.
         * 
         * @return  string
         */
        public function render()
        {
            $input = $this->getInput();

            if (!empty($this->args['hidden'])) {
                // we have an hidden field, only display the input
                return $input;
            }

            // render label only in case it hasn't been explicitly hidden
            if (empty($this->args['hiddenLabel'])) {
                $label = '<label for="' . ($this->args['id'] ?? '') . '">' . ($this->args['label'] ?? '') . '</label>';
            }

            // render description, only if provided
            if (!empty($this->args['description'])) {
                $help = '<p>' . $this->args['description'] . '</p>';
            }

            // create control
            return '<div class="form-field ' . $this->type . '-field">' . ($label ?? '') . $input . ($help ?? '') . '</div>';
        }

        /**
         * Renders the input of the form field.
         * 
         * @return  string
         */
        abstract protected function getInput();

        /**
         * @inheritDoc
         */
        #[\ReturnTypeWillChange]
        final public function __toString()
        {
            return $this->render();
        }
    }

    /**
     * Standard HTML input text field renderer class.
     */
    class TextField extends FormField
    {
        protected string $type = 'text';

        /**
         * @inheritDoc
         */
        protected function getInput()
        {
            $name     = (string) ($this->args['name']     ?? '');
            $id       = (string) ($this->args['id']       ?? '');
            $class    = (string) ($this->args['class']    ?? '');
            $hint     = (string) ($this->args['hint']     ?? '');
            $size     = (string) ($this->args['size']     ?? '');
            $value    = (string) ($this->args['value']    ?? '');
            $readonly = (bool)   ($this->args['readonly'] ?? false);
            $disabled = (bool)   ($this->args['disabled'] ?? false);

            return '<input
                type="text"'
                . ($name ? 'name="' . \esc_attr($name) . '"' : '')
                . ($id ? 'id="' . \esc_attr($id) . '"' : '')
                . ($class ? 'class="' . \esc_attr($class) . '"' : '')
                . ($value ? 'value="' . \esc_attr($value) . '"' : '')
                . ($size ? 'size="' . ((int) $size) . '"' : '')
                . ($hint ? 'placeholder="' . \esc_attr($hint) . '"' : '')
                . ($readonly ? 'readonly' : '')
                . ($disabled ? 'disabled' : '')
                . '/>';
        }
    }

    /**
     * Standard HTML input textarea field renderer class.
     */
    class TextareaField extends FormField
    {
        protected string $type = 'textarea';

        /**
         * @inheritDoc
         */
        protected function getInput()
        {
            $name     = (string) ($this->args['name']     ?? '');
            $id       = (string) ($this->args['id']       ?? '');
            $class    = (string) ($this->args['class']    ?? '');
            $hint     = (string) ($this->args['hint']     ?? '');
            $rows     = (string) ($this->args['rows']     ??  3);
            $value    = (string) ($this->args['value']    ?? '');
            $readonly = (bool)   ($this->args['readonly'] ?? false);
            $disabled = (bool)   ($this->args['disabled'] ?? false);

            return '<textarea '
                . ($name ? 'name="' . \esc_attr($name) . '"' : '')
                . ($id ? 'id="' . \esc_attr($id) . '"' : '')
                . ($class ? 'class="' . \esc_attr($class) . '"' : '')
                . ($rows ? 'rows="' . ((int) $rows) . '"' : '')
                . ($hint ? 'placeholder="' . \esc_attr($hint) . '"' : '')
                . ($readonly ? 'readonly' : '')
                . ($disabled ? 'disabled' : '')
                . '>' . \esc_html($value) . '</textarea>';
        }
    }

    /**
     * Standard HTML input checkbox field renderer class.
     */
    class CheckboxField extends FormField
    {
        protected string $type = 'checkbox';

        /**
         * @inheritDoc
         */
        protected function getInput()
        {
            $name     = (string) ($this->args['name']     ?? '');
            $id       = (string) ($this->args['id']       ?? '');
            $class    = (string) ($this->args['class']    ?? '');
            $value    = (string) ($this->args['value']    ??  1);
            $checked  = (bool)   ($this->args['checked'] ?? false);
            $disabled = (bool)   ($this->args['disabled'] ?? false);

            return '<input
                type="checkbox"'
                . ($name ? 'name="' . \esc_attr($name) . '"' : '')
                . ($id ? 'id="' . \esc_attr($id) . '"' : '')
                . ($class ? 'class="' . \esc_attr($class) . '"' : '')
                . ($value ? 'value="' . \esc_attr($value) . '"' : '')
                . ($checked ? 'checked' : '')
                . ($disabled ? 'disabled' : '')
                . '/>';
        }
    }

    /**
     * Standard HTML select field renderer class.
     */
    class SelectField extends FormField
    {
        protected string $type = 'select';

        /**
         * @inheritDoc
         */
        protected function getInput()
        {
            $name     = (string) ($this->args['name']     ?? '');
            $id       = (string) ($this->args['id']       ?? '');
            $class    = (string) ($this->args['class']    ?? '');
            $value    = (string) ($this->args['value']    ??  1);
            $multiple = (bool)   ($this->args['multiple'] ?? false);
            $readonly = (bool)   ($this->args['readonly'] ?? false);
            $disabled = (bool)   ($this->args['disabled'] ?? false);

            $options = [];

            foreach ((array) ($this->args['options'] ?? []) as $optVal => $optText) {
                // check whether the option has been selected
                $selected = (is_array($value) && in_array($optVal, $value)) || $optVal == $value;
                // register option tag
                $options[] = '<option value="' . \esc_attr($optVal) . '"' . ($selected ? ' selected' : '') . '>' . $optText . '</option>';
            }

            return '<select '
                . ($name ? 'name="' . \esc_attr($name) . '"' : '')
                . ($id ? 'id="' . \esc_attr($id) . '"' : '')
                . ($class ? 'class="' . \esc_attr($class) . '"' : '')
                . ($multiple ? 'multiple' : '')
                . ($disabled ? 'disabled' : '')
                . ($readonly ? 'readonly' : '')
                . '>' . implode("\n", $options) . '</select>';
        }
    }

}

/**
 * Namespace used to render plugin views.
 */
namespace VikWP\WhiteLabel\Views {

    /**
     * Thickbox abstract view.
     */
    abstract class Thickbox
    {
        /**
         * Renders the thickbox.
         * 
         * @return  void
         */
        public function render()
        {
            // add any registered inline style
            if ($style = $this->includeStyle()) {
                echo '<style>' . $style . '</style>';
            }

            // open thickbox wrapper
            echo '<div id="whitelabel-settings" style="display: none;">';

            // display modal body
            echo '<div class="th-box-body">' . $this->displayBody() . '</div>';

            // display modal footer, if any
            if ($footer = $this->displayFooter()) {
                echo '<div class="th-box-footer">' . $footer . '</div>';
            }

            // close thickbox modal
            echo '</div>';

            // add any registered inline script
            if ($script = $this->includeScript()) {
                echo '<script>' . $script . '</script>';
            }
        }

        /**
         * Returns the inline styles.
         * 
         * @return  string
         */
        protected function includeStyle()
        {
            return '';
        }

        /**
         * Returns the thickbox body.
         * 
         * @return  string
         */
        abstract protected function displayBody();

        /**
         * Returns the thickbox footer.
         * 
         * @return  string
         */
        protected function displayFooter()
        {
            return '';
        }

        /**
         * Returns the inline scripts.
         * 
         * @return  string
         */
        protected function includeScript()
        {
            return '';
        }
    }

    /**
     * Displays the thickbox to manage the plugin settings.
     */
    class SettingsThickbox extends Thickbox
    {
        /** @var VikWP\WhiteLabel\Models\SettingsModel */
        protected $model;

        /**
         * Class constructor.
         */
        public function __construct()
        {
            $this->model = new \VikWP\WhiteLabel\Models\SettingsModel;
        }

        /**
         * @inheritDoc
         */
        protected function includeStyle()
        {
            return <<<CSS
.thickbox-loading {
    height: auto !important;
}
#TB_window {
    height: calc(100vh - 60px) !important;
}
#TB_ajaxContent {
    width: calc(100% - 30px) !important;
    height: calc(100% - 45px) !important;
}
.th-box-body {
    height: calc(100% - 30px);
    overflow-x: scroll;
}
.th-box-footer {
    text-align: right;
}
.form-wrap .form-field.checkbox-field label {
    display: inline-block;
    margin-right: 10px;
}
CSS;
        }

        /**
         * @inheritDoc 
         */
        protected function displayBody()
        {
            $panels = [];

            // add company name panel
            $panels[] = $this->displayCompanySettings();

            // add VikAppointments panel, only if installed
            if (is_file(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'vikappointments' . DIRECTORY_SEPARATOR . 'vikappointments.php')) {
                $panels[] = $this->displayVikAppointmentsSettings();
            }

            // add VikBooking panel, only if installed
            if (is_file(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'vikbooking' . DIRECTORY_SEPARATOR . 'vikbooking.php')) {
                $panels[] = $this->displayVikBookingSettings();
            }

            // add VikRentCar panel, only if installed
            if (is_file(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'vikrentcar' . DIRECTORY_SEPARATOR . 'vikrentcar.php')) {
                $panels[] = $this->displayVikRentCarSettings();
            }

            // add VikRentItems panel, only if installed
            if (is_file(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'vikrentitems' . DIRECTORY_SEPARATOR . 'vikrentitems.php')) {
                $panels[] = $this->displayVikRentItemsSettings();
            }

            // add VikRestaurants panel, only if installed
            if (is_file(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'vikrestaurants' . DIRECTORY_SEPARATOR . 'vikrestaurants.php')) {
                $panels[] = $this->displayVikRestaurantsSettings();
            }

            // adds a notice at the end of the modal to instruct the administrator on how to disable this plugin
            $panels[] = '<hr /><div class="notice notice-warning inline" style="margin: 25px 0;"><p>If you want to disable the white labelling applied by this plugin, you need to delete the file below via FTP.</p><pre><code>' . __FILE__ . '</code></pre></div>';

            // marge all the panels together
            return '<form id="whitelabel-form">' . implode("\n", $panels) . '</form>';
        }

        /**
         * Displays the company settings.
         * 
         * @return string
         */
        protected function displayCompanySettings()
        {
            return new \VikWP\WhiteLabel\Form\FormFieldset('Company Name', [
                // company name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[company][name]',
                    'label' => 'Company Name',
                    'description' => 'Overwrite the default company name of the plugin manufacturer.',
                    'value' => $this->model->get('company.name', ''),
                ]),
                // company URL
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[company][url]',
                    'label' => 'Company URL',
                    'description' => 'Insert here the URL where the company name should point (plugins list).',
                    'value' => $this->model->get('company.url', ''),
                ]),
                // plugin name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[company][plugin][name]',
                    'label' => 'Plugin Name',
                    'description' => 'Changes the name of this plugin from "' . $this->model->get('company.plugin.name', 'White Label') . '" into the value specified here.',
                    'value' => $this->model->get('company.plugin.name', ''),
                ]),
                // plugin description
                new \VikWP\WhiteLabel\Form\TextareaField([
                    'name' => 'whitelabel[company][plugin][description]',
                    'label' => 'Plugin Description',
                    'description' => 'Specify an optional description for this plugin, if you wish.',
                    'value' => $this->model->get('company.plugin.description', ''),
                ]),
            ]);
        }

        /**
         * Displays the settings for VikAppointments.
         * 
         * @return  string
         */
        protected function displayVikAppointmentsSettings()
        {
            return new \VikWP\WhiteLabel\Form\FormFieldset('VikAppointments', [
                // plugin name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[vikappointments][name]',
                    'label' => 'Plugin Name',
                    'description' => 'Replace any occurrences of "VikAppointments" with the text specified here.',
                    'value' => $this->model->get('vikappointments.name', ''),
                ]),
                // enable updates
                new \VikWP\WhiteLabel\Form\CheckboxField([
                    'name' => 'whitelabel[vikappointments][updates]',
                    'label' => 'Enable Updates',
                    'description' => 'Choose whether the updates of VikAppointments should be visible. Only you will be able to see the available updates.',
                    'checked' => (bool) $this->model->get('vikappointments.updates', false),
                ]),
            ]);
        }

        /**
         * Displays the settings for VikBooking.
         * 
         * @return  string
         */
        protected function displayVikBookingSettings()
        {
            return new \VikWP\WhiteLabel\Form\FormFieldset('VikBooking', [
                // plugin name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[vikbooking][name]',
                    'label' => 'Plugin Name',
                    'description' => 'Replace any occurrences of "VikBooking" with the text specified here.',
                    'value' => $this->model->get('vikbooking.name', ''),
                ]),
                // enable updates
                new \VikWP\WhiteLabel\Form\CheckboxField([
                    'name' => 'whitelabel[vikbooking][updates]',
                    'label' => 'Enable Updates',
                    'description' => 'Choose whether the updates of VikBooking should be visible. Only you will be able to see the available updates.',
                    'checked' => (bool) $this->model->get('vikbooking.updates', false),
                ]),
            ]);
        }

        /**
         * Displays the settings for VikRentCar.
         * 
         * @return  string
         */
        protected function displayVikRentCarSettings()
        {
            return new \VikWP\WhiteLabel\Form\FormFieldset('VikRentCar', [
                // plugin name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[vikrentcar][name]',
                    'label' => 'Plugin Name',
                    'description' => 'Replace any occurrences of "VikRentCar" with the text specified here.',
                    'value' => $this->model->get('vikrentcar.name', ''),
                ]),
                // enable updates
                new \VikWP\WhiteLabel\Form\CheckboxField([
                    'name' => 'whitelabel[vikrentcar][updates]',
                    'label' => 'Enable Updates',
                    'description' => 'Choose whether the updates of VikRentCar should be visible. Only you will be able to see the available updates.',
                    'checked' => (bool) $this->model->get('vikrentcar.updates', false),
                ]),
            ]);
        }

        /**
         * Displays the settings for VikRentItems.
         * 
         * @return  string
         */
        protected function displayVikRentItemsSettings()
        {
            return new \VikWP\WhiteLabel\Form\FormFieldset('VikRentItems', [
                // plugin name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[vikrentitems][name]',
                    'label' => 'Plugin Name',
                    'description' => 'Replace any occurrences of "VikRentItems" with the text specified here.',
                    'value' => $this->model->get('vikrentitems.name', ''),
                ]),
                // enable updates
                new \VikWP\WhiteLabel\Form\CheckboxField([
                    'name' => 'whitelabel[vikrentitems][updates]',
                    'label' => 'Enable Updates',
                    'description' => 'Choose whether the updates of VikRentItems should be visible. Only you will be able to see the available updates.',
                    'checked' => (bool) $this->model->get('vikrentitems.updates', false),
                ]),
            ]);
        }

        /**
         * Displays the settings for VikRestaurants.
         * 
         * @return  string
         */
        protected function displayVikRestaurantsSettings()
        {
            return new \VikWP\WhiteLabel\Form\FormFieldset('VikRestaurants', [
                // plugin name
                new \VikWP\WhiteLabel\Form\TextField([
                    'name' => 'whitelabel[vikrestaurants][name]',
                    'label' => 'Plugin Name',
                    'description' => 'Replace any occurrences of "VikRestaurants" with the text specified here.',
                    'value' => $this->model->get('vikrestaurants.name', ''),
                ]),
                // enable updates
                new \VikWP\WhiteLabel\Form\CheckboxField([
                    'name' => 'whitelabel[vikrestaurants][updates]',
                    'label' => 'Enable Updates',
                    'description' => 'Choose whether the updates of VikRestaurants should be visible. Only you will be able to see the available updates.',
                    'checked' => (bool) $this->model->get('vikrestaurants.updates', false),
                ]),
            ]);
        }

        /**
         * @inheritDoc 
         */
        protected function displayFooter() {
            $btnSaveText = __('Save');

            return <<<HTML
<a href="javascript:void(0)" id="whitelabel-settings-save" class="button button-primary">{$btnSaveText}</a>
HTML;
        }

        /**
         * @inheritDoc
         */
        protected function includeScript() {
            return <<<JS
(function($) {
    'use strict';

    const saveSettings = (data) => {
        return new Promise((resolve, reject) => {
            $.ajax({
                url: 'admin-ajax.php?action=whitelabel_settings_save',
                type: 'post',
                data: data,
            }).done(() => {
                resolve();
            }).fail((error) => {
                reject(error.responseText || error.statusText || 'An error has occurred! Please try again.');
            });
        });
    } 

    $(function() {
        $('#whitelabel-settings-save').on('click', async function() {
            if ($(this).hasClass('disabled')) {
                return false;
            }

            $(this).addClass('disabled');

            const data = $('#whitelabel-form').serialize();
            
            try {
                await saveSettings(data);
                // auto-dispose the thickbox
                $('#TB_closeWindowButton').trigger('click');
                // refresh the page after a short delay
                setTimeout(() => { document.location.reload() }, 1024);
            } catch (err) {
                try {
                    err = JSON.parse(err).data;
                } catch (malformed) {
                    // do nothing
                }

                alert(err);
            }

            $(this).removeClass('disabled');            
        });
    });
})(jQuery);
JS;
        }
    }

}

/**
 * Namespace used to define the models that will interacts among
 * the views and the controllers.
 */
namespace VikWP\WhiteLabel\Models {

    /**
     * White label settings model.
     */
    class SettingsModel
    {
        /** @var array */
        protected array $settings;

        /**
         * Class constructor.
         */
        public function __construct()
        {
            // preload the saved settings
            $this->settings = json_decode(\get_option('vikwp_white_label_settings', '{}'), true);
        }

        /**
         * Returns the value configured for the specified key.
         * Supports a dotted-notation to access nested attributes.
         * 
         * @param   string  $key      The name of the parameter to access.
         * @param   mixed   $default  The default value in case the parameter hasn't been configured.
         */
        public function get(?string $key = null, $default = null)
        {
            if (!$key) {
                return $this->settings;
            }

            $value = $this->settings;

            foreach (explode('.', $key) as $chunk) {
                if (!is_array($value)) {
                    return $default;
                }

                if (!isset($value[$chunk])) {
                    return $default;
                }

                $value = $value[$chunk];
            }

            return $value === '' || $value === null ? $default : $value;
        }

        /**
         * Saves the settings.
         * 
         * @param   array  $data
         * 
         * @return  void
         * 
         * @throws  \Exception
         */
        public function save(array $data)
        {
            // merge the existing settings with the provided ones
            $this->settings = array_merge($this->settings, $data);

            // save the settings
            \update_option('vikwp_white_label_settings', json_encode($this->settings));
        }
    }

}

/**
 * Namespace used to handle the requests made through the views.
 */
namespace VikWP\WhiteLabel\Controllers {

    /**
     * AJAX end-point used to save the settings of the plugin.
     * 
     * @return  void
     */
    \add_action('wp_ajax_whitelabel_settings_save', function() {
        // extract the settings from the request
        $settings = $_REQUEST['whitelabel'] ?? [];

        try {
            // save the settings
            (new \VikWP\WhiteLabel\Models\SettingsModel)->save($settings);
        } catch (\Throwable $error) {
            // an error has occurred
            \wp_send_json_error($error->getMessage(), $error->getCode() ?: 500);
        }

        // settings saved successfully
        \wp_send_json_success();
    });

}
