<?php
/*
Plugin Name:  White-Label Installer
Description:  Installer of the White-Label must-use plugin. This plugin is automatically uninstalled upon activation.
Version:      1.0.1
Author:       E4J s.r.l.
Author URI:   https://vikwp.com
License:      GPL2
License URI:  https://www.gnu.org/licenses/gpl-2.0.html
*/

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

/**
 * Callback used to handle the activation process.
 * 
 * @return  void
 */
function whitelabel_installer_activation()
{
    try {
        // save the ID of the user that should be allowed to manage the settings of the "White Label" plugin
        update_option('whitelabel_manager_user_id', wp_get_current_user()->ID);

        // check whether this website defines a custom folder for the MUST-USE plugins
        if (defined('WPMU_PLUGIN_DIR')) {
            // use the custom folder (remove trailing slash)
            $mustUsePluginsPath = untrailingslashit(WPMU_PLUGIN_DIR);
        } else {
            // use the default folder
            $mustUsePluginsPath = ABSPATH . 'wp-content/mu-plugins';
        }

        global $wp_filesystem;

        if (!function_exists('WP_Filesystem')) {
            // manually load the file system dependencies
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        if (!$wp_filesystem) {
            // boot the file system here
            WP_Filesystem();
        }

        // check whether the must-use folder exists
        if (!$wp_filesystem->exists($mustUsePluginsPath)) {
            if (!$wp_filesystem->mkdir($mustUsePluginsPath, FS_CHMOD_DIR)) {
                // unable to create the folder
                throw new RuntimeException("<p>Unable to create the following folder, please proceed manually.</p><pre><code>{$mustUsePluginsPath}</code></pre>", 500);
            }
        }

        // make sure the directory is writable
        if (!$wp_filesystem->is_writable($mustUsePluginsPath)) {
            // folder not writable
            throw new RuntimeException("<p>The folder where the <strong>must-use</strong> plugin should be copied is not writable. Please manually adjust the permissions for the following folder.</p><pre><code>{$mustUsePluginsPath}</code></pre>", 500);
        }

        // create source and destination paths
        $srcPath  = dirname(__FILE__) . '/must-use/whitelabel.php';
        $destPath = $mustUsePluginsPath . '/whitelabel.php';

        // copy the source file into the correct path
        if (!$wp_filesystem->copy($srcPath, $destPath, $overwrite = true)) {
            // copy failed
            throw new RuntimeException("<p>Unable to copy the <strong>white label</strong> plugin into the following folder, please proceed manually.</p><pre><code>$mustUsePluginsPath</code></pre>", 500);
        }
    } catch (Throwable $error) {
        // display the error message
        wp_die($error->getMessage(), 'Plugin activation failed', [
            'code' => $error->getCode() ?: 500,
            'back_link' => true,
        ]);
    }
}

// handle plugin activation
register_activation_hook(__FILE__, 'whitelabel_installer_activation');

/**
 * Auto-delete this plugin as soon as it is active.
 */
add_action('init', function() {

    // manually load all the dependencies
    require_once ABSPATH . 'wp-admin/includes/file.php';
    require_once ABSPATH . 'wp-admin/includes/plugin.php';

    // auto-delete the plugin upon completion
    delete_plugins([
        plugin_basename(__FILE__),
    ]);

    // redirect to the must-use plugin
    $landingPage = 'plugins.php?plugin_status=mustuse';

    // redirect is allowed only if the headers haven't been sent yet
    if (!headers_sent()) {
        // perform an HTTP redirect
        wp_redirect($landingPage);
    } else {
        // otherwise redirect using javascript
        echo "<script>document.location.href='{$landingPage}';</script>\n";
    }
});
