<?php
/** 
 * @package   	VikReCAPTCHA
 * @subpackage 	core
 * @author    	E4J s.r.l.
 * @copyright 	Copyright (C) 2019 E4J s.r.l. All Rights Reserved.
 * @license  	http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link 		https://vikwp.com
 */

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

/**
 * ReCAPTCHA default handler.
 *
 * @since 1.0
 */
class VikRecaptchaDriverDefault implements VikRecaptchaDriver
{
	/**
	 * The configuration array.
	 *
	 * @var array
	 */
	protected $options;

	/**
	 * Class constructor.
	 *
	 * @param 	array 	$options 	An array of options.
	 */
	public function __construct(array $options = array())
	{
		// load configuration
		$this->options = array();
		$this->options['version']   = get_option('vikrecaptcha_version',   '2.0');
		$this->options['publickey'] = get_option('vikrecaptcha_public_key', null);
		$this->options['secretkey'] = get_option('vikrecaptcha_secret_key', null);
		$this->options['theme']     = get_option('vikrecaptcha_theme', 'light');
		$this->options['size']      = get_option('vikrecaptcha_size', 'normal');

		// merge config with $options
		$this->options = array_merge($this->options, $options);
	}

	/**
	 * Displays the ReCAPTCHA element.
	 *
	 * @return 	string 	The HTML to display.
	 */
	public function display()
	{
		$dom = new DOMDocument('1.0', 'UTF-8');

		/**
		 * Use dynamic ID to support multiple ReCaptcha within the same pages.
		 *
		 * @since 1.1
		 */
		static $counter = 0;

		if (!empty($this->options['id']))
		{
			// use given ID
			$id = $this->options['id'];
		}
		else
		{
			// generate a random one
			$id = 'vik_dynamic_recaptcha_' . (++$id);
		}

		$el = $dom->createElement('div');
		$el->setAttribute('id', $id);

		// display ReCAPTCHA v2.0
		if ($this->options['version'] == '2.0')
		{
			// load Google ReCAPTCHA v2.0
			wp_enqueue_script('vikrecaptcha', 'https://www.google.com/recaptcha/api.js?hl=' . get_locale());

			$el->setAttribute('class', 'g-recaptcha');
			$el->setAttribute('data-sitekey', $this->options['publickey']);
			$el->setAttribute('data-theme', $this->options['theme']);
			$el->setAttribute('data-size', $this->options['size']);
			$el->setAttribute('data-tabindex', '0');
			$el->setAttribute('data-callback', '');
			$el->setAttribute('data-expired-callback', '');
			$el->setAttribute('data-error-callback', '');
		}

		$dom->appendChild($el);
		return $dom->saveHTML($el);
	}

	/**
	 * Checks whether ReCAPTCHA has been submitted properly.
	 *
	 * @return 	boolean  True if correct, false otherwise.
	 *
	 * @throws  Exception
	 */
	public function check()
	{
		// obtain IP address
		$ipaddr = $_SERVER['REMOTE_ADDR'];

		if (!$ipaddr)
		{
			// invalid IP
			throw new RuntimeException(__('For security reasons, you must pass the remote IP address to reCAPTCHA.', 'vikrecaptcha'), 400);
		}

		// extract token
		$token = preg_replace("/[^a-zA-Z0-9_-]+/", '', $_REQUEST['g-recaptcha-response']);

		if (!$token)
		{
			// possible SPAM attempt
			throw new RuntimeException(__('Please complete the CAPTCHA.', 'vikrecaptcha'), 400);
		}

		// validate ReCAPTCHA v2.0
		if ($this->options['version'] == '2.0')
		{
			// validation end-points
			$url = 'https://www.google.com/recaptcha/api/siteverify';

			$args = array(
				'body' => array(
					'secret'   => $this->options['secretkey'],
					'response' => $token,
					'remoteip' => $ipaddr,
				),
			);

			// make connection with Google
			$response = wp_remote_post($url, $args);

			if (is_wp_error($response))
			{
				// an error occurred
   				throw new Exception($response->get_error_message());
			}
			else if ($response['response']['code'] == 200)
			{
				// decode BODY
				$body = (array) json_decode($response['body'], true);

				if (isset($body['success']) && $body['success'] == true)
				{
					// ReCAPTCHA has been validated properly
					return true;
				}

				// fetch error codes
				$err = !empty($body['error-codes']) ? implode(', ', $body['error-codes']) : '<pre>' . print_r($body, true) . '</pre>';

				// raise detailed error
				throw new Exception(sprintf(__('An error occurred while validating the CAPTCHA: %s', 'vikrecaptcha'), $err));
			}
			else
			{
				throw new Exception(__('It was not possible to reach Google for CAPTCHA validation.', 'vikrecaptcha'));
			}
		}

		// missing implementation for the specified version
		throw new RuntimeException(sprintf(__('Missing [%s %s] implementation.', 'vikrecaptcha'), __CLASS__, $this->options['version']), 501);
	}

	/**
	 * Checks whether this driver can be used.
	 *
	 * @return 	boolean  True if usable, false otherwise.
	 */
	public function isSupported()
	{
		// make sure the public key and secret key have been specified
		return $this->options['publickey'] && $this->options['secretkey'];
	}
}
