<?php
/** 
 * @package   	VikReCAPTCHA
 * @subpackage 	core
 * @author    	E4J s.r.l.
 * @copyright 	Copyright (C) 2019 E4J s.r.l. All Rights Reserved.
 * @license  	http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link 		https://vikwp.com
 */

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

require_once dirname(__FILE__) . DIRECTORY_SEPARATOR . 'driver.php';

/**
 * Factory class used to access drivers able to handle ReCAPTCHA plugin
 * provided by Google.
 *
 * @since 1.0
 */
class VikRecaptchaFactory
{
	/**
	 * The cached instance.
	 *
	 * @var VikRecaptchaDriver
	 */
	protected static $instance = false;

	/**
	 * Returns the most appropriate ReCAPTCHA driver.
	 *
	 * @param 	array   $options  A configuration array. 
	 *
	 * @return 	mixed 	The driver, false otherwise.
	 */
	public static function getDriver(array $options = array())
	{
		// search for a cached version
		if (static::$instance !== false)
		{
			return static::$instance;
		}

		static::$instance = null;

		// base path
		$base = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'drivers';

		// find all the existing drivers
		$drivers = glob($base . DIRECTORY_SEPARATOR . '*.php');
		// extract driver names
		$drivers = array_map(function($el)
		{
			if (preg_match("/(.*?)\.php$/", basename($el), $match))
			{
				$el = end($match);
			}

			return $el;
		}, $drivers);

		// find default driver
		$index = array_search('default', $drivers);

		if ($index !== false && end($drivers) !== 'default')
		{
			// remove default driver
			array_splice($drivers, $index, 1);
			// push default driver as last element
			$drivers[] = 'default';
		}

		/**
		 * Filter used to manipulate options array at runtime before
		 * it is used by the driver found, if any.
		 *
		 * @param 	array 	$options  A configuration array.
		 *
		 * @return 	array 	The configuration array.
		 */
		$options = (array) apply_filters('vik_recaptcha_driver_options', $options);

		// iterate drivers until we find one available
		while (!static::$instance && ($driver = array_shift($drivers)))
		{
			// include file
			require_once $base . DIRECTORY_SEPARATOR . $driver . '.php';

			// fetch class name
			$classname = 'VikRecaptchaDriver' . ucfirst($driver);

			// make sure the class exists
			if (class_exists($classname))
			{
				// instantiate driver
				$obj = new $classname($options);

				// check if the classname if a valid instance and is supported
				if ($obj instanceof VikRecaptchaDriver && $obj->isSupported())
				{
					// instantiate and cache
					static::$instance = $obj;
				}
			}
			else if (WP_DEBUG)
			{
				// in case DEBUG is enabled, throw exception
				throw new Exception(sprintf(__('Class [%s] not found'), $classname), 404);
			}
		}

		return static::$instance;
	}
}
