<?php
/*
Plugin Name:  VikReCAPTCHA
Plugin URI:   https://vikwp.com/
Description:  Bridge used to support reCAPTCHA Google module on any VIK plugins.
Version:      1.2
Author:       E4J s.r.l.
Author URI:   https://vikwp.com
License:      GPL2
License URI:  https://www.gnu.org/licenses/gpl-2.0.html
Text Domain:  vikrecaptcha
Domain Path:  /languages
*/

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

define('VIKRECAPTCHA_SOFTWARE_VERSION', '1.2');
define('VIKRECAPTCHA_PATH', __FILE__);

// load reCAPTCHA factory
require_once implode(DIRECTORY_SEPARATOR, [dirname(__FILE__), 'libraries', 'recaptcha', 'factory.php']);
require_once implode(DIRECTORY_SEPARATOR, [dirname(__FILE__), 'libraries', 'feedback', 'handler.php']);

/**
 * Displays the reCAPTCHA plugin by appending the
 * HTML content within the $response argument.
 *
 * @param 	string 	&$response 	The HTML to display.
 * @param 	array 	$options 	An array of options.
 *
 * @return 	void
 *
 * @throws 	Exception
 *
 * @since 	1.0
 */
add_action('vik_recaptcha_display', function(&$response, array $options = [])
{
	// go ahead only in case none else manipulated the response
	if ($response === null) {
		// get driver
		$driver = VikRecaptchaFactory::getDriver($options);

		if ($driver) {
			try {
				// catch displayed reCAPTCHA HTML
				$response = $driver->display();
			} catch (Exception $e) {
				// something went wrong with the driver
				if (WP_DEBUG) {
					// re-throw exception only in case DEBUG is enabled
					throw $e;
				} else {
					// invalidate response
					$response = '';
				}
			}
		}
	}
}, 90, 2);

/**
 * Checks whether the reCAPTCHA has been submitted properly.
 *
 * @param 	boolean  &$response  True if valid.
 * @param 	array 	 $options 	 An array of options.
 *
 * @return 	void
 *
 * @throws 	Exception
 *
 * @since 	1.0
 */
add_action('vik_recaptcha_check', function(&$response, array $options = []) {
	// go ahead only in case none else manipulated the response
	if ($response === null) {
		// get driver
		$driver = VikRecaptchaFactory::getDriver($options);

		if ($driver) {
			try {
				// try validating the token
				$response = $driver->check();
			} catch (Exception $e) {
				// something went wrong with the validation
				if (WP_DEBUG) {
					// re-throw exception only in case DEBUG is enabled
					throw $e;
				} else {
					// invalidate response
					$response = false;
				}
			}
		}
	}
}, 90, 2);

/**
 * Checks whether the plugin reCAPTCHA can be used.
 *
 * @param  bool  $on  The current plugin status (false by default).
 *
 * @since  1.0
 */
add_filter('vik_recaptcha_on', function($on = false)
{
	return $on || VikRecaptchaFactory::getDriver() !== null;
});

/**
 * Checks whether the plugin reCAPTCHA cannot be used.
 *
 * @param  bool  $on  The current plugin status (false by default).
 *
 * @since  1.0
 */
add_filter('vik_recaptcha_off', function($on = false) {
	return !$on && VikRecaptchaFactory::getDriver() === null;
});

///////////////////////////////////////////////////////////////////////
///////////////////////// CUSTOM PLUGIN HOOKS /////////////////////////
///////////////////////////////////////////////////////////////////////

/**
 * Google Captcha (reCAPTCHA) by BestWebSoft plugin works only if
 * the custom plugin is enabled from the back-end. So, we should use
 * the hook below in order to start supporting VikRecaptcha.
 *
 * Then, the plugin can be found under:
 * Google Captcha > Enable reCaptcha for > Custom Forms
 *
 * @param 	array 	$forms 	An array containing all the form options for custom plugins.
 *
 * @return 	array 	The updated forms.
 *
 * @since 	1.0
 */
add_filter('gglcptch_add_custom_form', function($forms) {
	$forms['vikrecaptcha'] = [
		'form_name' => 'VikReCAPTCHA',
	];

    return $forms;
});

/**
 * Filters the action links displayed for each plugin in the Plugins list table.
 *
 * @param 	array   $actions     An array of plugin action links. By default this can include 'activate',
 *                               'deactivate', and 'delete'. With Multisite active this can also include
 *                               'network_active' and 'network_only' items.
 * @param 	string  $plugin_file  Path to the plugin file relative to the plugins directory.
 * @param 	array   $plugin_data  An array of plugin data. See `get_plugin_data()`.
 * @param 	string  $context      The plugin context. By default this can include 'all', 'active', 'inactive',
 *                               'recently_activated', 'upgrade', 'mustuse', 'dropins', and 'search'.
 *
 * @since 	2.5.0
 * @since 	2.6.0   The `$context` parameter was added.
 * @since 	4.9.0   The 'Edit' link was removed from the list of action links.
 */
add_filter('plugin_action_links', ['VikFeedbackHandler', 'deactivate'], 10, 4);

/**
 * Fires authenticated Ajax actions for logged-in users.
 *
 * The dynamic portion of the hook name, `$action`, refers
 * to the name of the Ajax action callback being fired.
 *
 * @since 2.1.0
 */
add_action('wp_ajax_vikrecaptcha', function() {
	if (isset($_REQUEST['task']) && $_REQUEST['task'] == 'feedback') {
		// perform notification
		VikFeedbackHandler::notify();
	}

	// die to get a valid response
	wp_die();
});

///////////////////////////////////////
//////// PLUGIN UPDATE CHANNEL ////////
///////////////////////////////////////

/**
 * Subscribes the plugin to the update channel provided by VikUpdater 2.0.
 */
add_filter('vikupdater_subscribe_plugins', function($subscribers) {
    $subscribers[] = [
        // this is the SKU
        'sku' => 'captcha',
        // this is the plugin name (supports i18n)
        'name' => __('VikReCAPTCHA', 'vikrecaptcha'),
        // this is the base name of the plugin main file (without ".php")
        'slug' => 'vikrecaptcha',
        // this is usually built as "[FOLDER_NAME]/[MAIN_FILE_NAME].php"
        'plugin' => 'vikrecaptcha/vikrecaptcha.php',
        // the currently installed version of the plugin (possibly defined by a constant)
        'version' => VIKRECAPTCHA_SOFTWARE_VERSION,
        // the minimum required version of WordPress (optional)
        'requires' => '4.0',
        // the minimum required version of PHP (optional)
        'requires_php' => '7.0',
        // the short description of the plugin (supports i18n)
        'description' => __('Bridge used to support reCAPTCHA Google module on any VIK plugins.', 'vikrecaptcha'),
        // specify the URL where the latest version of the plugin should be downloaded
        'download' => 'https://vikwp.com/api/?task=products.freedownload&sku=captcha',
    ];

    return $subscribers;
});

/**
 * Registers the program icons that will be used by WordPress when a new version is detected.
 */
add_filter('vikupdater_prepare_update_plugin_data', function($update, $manifest, $options) {
    // make sure we are observing our plugin (same as the "slug" previously registered)
    if ($update->slug === 'vikrecaptcha') {
        // register the plugin icons
        // $update->icons = [
        //     '2x' => plugin_dir_url(__FILE__) . 'assets/images/logo-256x256.jpg',
        //     '1x' => plugin_dir_url(__FILE__) . 'assets/images/logo-128x128.jpg',
        // ];
    }

    return $update;
}, 10, 3);

/**
 * Filters the array of row meta for each plugin in the Plugins list table.
 *
 * @param  string[]  $plugin_meta  An array of the plugin's metadata,
 *                                 including the version, author,
 *                                 author URI, and plugin URI.
 * @param  string    $plugin_file  Path to the plugin file relative to the plugins directory.
 * @param  array     $plugin_data  An array of plugin data.
 * @param  string    $status       Status of the plugin. Defaults are 'All', 'Active',
 *                                 'Inactive', 'Recently Activated', 'Upgrade', 'Must-Use',
 *                                 'Drop-ins', 'Search', 'Paused'.
 *
 * @since 	2.8.0
 */
add_filter('plugin_row_meta', function($plugin_meta, $plugin_file, $plugin_data, $status) {
	if ($plugin_file == 'vikrecaptcha/vikrecaptcha.php') {
		// URI to readme.html file
		$url = plugin_dir_url(__FILE__) . 'readme.html?TB_iframe=true';

		// add documentation link
		$plugin_meta[] = sprintf(
			'<a href="%s" class="thickbox open-plugin-details-modal" aria-label="%s" data-title="%s">%s</a>',
			esc_attr($url),
			esc_attr(sprintf(__('More information about %s'), 'VikReCAPTCHA')),
			esc_attr('VikReCAPTCHA'),
			__('Documentation')
		);
	}

	return $plugin_meta;
}, 10, 4);
